#pragma once

#include "context.h"
#include "logger.h"
#include "mdb_module.h"
#include "sharpei_client.h"
#include "macs_log.h"
#include "macs_profiler.h"
#include "save_envelope_op.h"
#include "update_envelope_op.h"
#include "store_mailish_op.h"

#include <mail/library/utf8/utf8.h>
#include <macs_pg/macs_pg.h>
#include <macs_pg/service/factory.h>
#include <pgg/chrono.h>
#include <pgg/database/pool_factory.h>
#include <pgg/profiling.h>
#include <sharpei_client/sharpei_client.h>
#include <user_journal/parameters/parameter_id.h>
#include <user_journal/service_factory.h>
#include <pa/async.h>
#include <yplatform/module.h>
#include <yplatform/coroutine.h>

#include <util/datetime/base.h>

namespace NMdb {

class TMdbModuleImpl: public TMdbModule, public yplatform::module {
public:
    void Save(
        TContextPtr context,
        const std::string& sessionId,
        const TUserInfo& userInfo,
        const TMessageBase& message,
        const TFolderCoords& destFolder,
        const TLabelsArgs& labels,
        const TMaybeFail<TTab>& tab,
        const THeaders& headers,
        const std::vector<TAttachment>& attachments,
        const std::vector<TMimePart>& parts,
        const TThreadInfo& threadInfo,
        const TStoreParams& params,
        ENoSuchFolderAction noSuchFolderAction,
        const TMaybeFail<TFilterActions>& filterActions,
        const THandler& handler) override
    {
        auto db = GetService(context, sessionId, userInfo);
        auto saveHandler =
            [this, sessionId, userInfo, filterActions, handler](boost::system::error_code ec, const TStoreResult& res) {
                if (!ec && filterActions) {
                    LogSpamReport(sessionId, userInfo, *filterActions, res);
                }
                handler(ec, res);
            };
        auto saveOp = std::make_shared<TSaveEnvelopeOp>(
            context,
            db,
            message,
            destFolder,
            labels,
            tab,
            headers,
            attachments,
            parts,
            threadInfo,
            params,
            noSuchFolderAction,
            SpamReportLogger ? filterActions : Nothing(),
            saveHandler);
        yplatform::spawn(saveOp);
    }

    void Update(
        TContextPtr context,
        const std::string& sessionId,
        const TUserInfo& userInfo,
        const std::string& oldMid,
        const TMessageBase& message,
        const std::string& fid,
        const TLabelsArgs& labels,
        const THeaders& headers,
        const std::vector<TAttachment>& attachments,
        const std::vector<TMimePart>& parts,
        const TThreadInfo& threadInfo,
        const TStoreParams& params,
        const THandler& handler) override
    {
        auto db = GetService(context, sessionId, userInfo);
        auto updateOp = std::make_shared<TUpdateEnvelopeOp>(
            context,
            db,
            oldMid,
            message,
            fid,
            labels,
            headers,
            attachments,
            parts,
            threadInfo,
            params,
            handler);
        yplatform::spawn(updateOp);
    }

    void StoreMailish(
        TContextPtr context,
        const std::string& sessionId,
        const TUserInfo& userInfo,
        const TMessageBase& message,
        const std::string& fid,
        const TLabelsArgs& labels,
        const TMaybeFail<TTab>& tab,
        const THeaders& headers,
        const std::vector<TAttachment>& attachments,
        const std::vector<TMimePart>& parts,
        const TThreadInfo& threadInfo,
        const TStoreParams& params,
        uint64_t externalImapId,
        const THandler& handler) override
    {
        auto db = GetService(context, sessionId, userInfo);
        auto storeOp = std::make_shared<TStoreMailishOp>(
            context,
            db,
            message,
            fid,
            labels,
            tab,
            headers,
            attachments,
            parts,
            threadInfo,
            params,
            externalImapId,
            handler);
        yplatform::spawn(storeOp);
    }

    macs::ServicePtr GetService(
        TContextPtr context,
        const std::string& sessionId,
        const TUserInfo& userInfo) override
    {
        auto resolverFactory = macs::pg::createSharpeiUidResolverFactory({
            std::make_shared<TSharpeiHttpClient>(context),
            SharpeiSettings,
            Credentials});

        macs::pg::RequestInfo requestInfo;
        requestInfo.clientType = "fastsrv";
        requestInfo.userIp = userInfo.IpFrom;

        auto factory = macs::pg::createSeviceFactory(ConnectionPool, resolverFactory);

        factory->queryConf(QueryConf);
        factory->credentials(Credentials);
        factory->autoQueryHandleStartegy(macs::pg::masterOnly);
        factory->logger(boost::make_shared<TMacsLog>(context));
        factory->transactionTimeout(TransactionTimeout);
        factory->requestInfo(requestInfo);
        factory->profiler(boost::make_shared<TMacsProfiler>(context));

        if (UserJournalService) {
            using namespace user_journal::parameters;
            auto userJournal = UserJournalService->createJournal(
                userInfo.Uid,
                std::make_tuple(
                    id::module("fastsrv"),
                    id::suid(userInfo.Suid),
                    id::ip(userInfo.IpFrom.empty() ? "127.0.0.1" : userInfo.IpFrom),
                    id::connectionId(sessionId.empty() ? context->uniq_id() : sessionId)));
            factory->userJournal(userJournal);
        }
        return factory->product(userInfo.Uid);
    }


    macs::pg::ConnectionPoolPtr GetConnectionPool() override {
        return ConnectionPool;
    }

    void init(const yplatform::ptree& conf) {
        InitPg(conf.get_child("pg"));
        InitSharpei(conf.get_child("sharpei"));
        InitUserJournal(conf.get_child("user_journal"));
        InitSpamReportLogger(conf);
    }

private:
    void InitPg(const yplatform::ptree& conf) {
        auto reactor = yplatform::find_reactor(conf.get<std::string>("reactor"));
        if (reactor->size() != 1) {
            throw std::runtime_error("macs::pg is optimized for single-pool reactors - set pool_count=1 and io_threads=N");
        }

        Credentials = {conf.get<std::string>("user"), ""};
        QueryConf = macs::pg::readQueryConfFile(conf.get<std::string>("query_conf"));
        TransactionTimeout = macs::pg::Milliseconds(conf.get<int>("transaction_timeout"));
        ConnectionPool = macs::pg::ConnectionPoolFactory()
            .ioService(*(reactor->io()))
            .connectTimeout(macs::pg::Milliseconds(conf.get<int>("connect_timeout")))
            .queryTimeout(macs::pg::Milliseconds(conf.get<int>("query_timeout")))
            .queueTimeout(macs::pg::Milliseconds(conf.get<int>("queue_timeout")))
            .maxConnections(conf.get<int>("max_connections"))
            .asyncResolve(conf.get<bool>("async_resolve"))
            .ipv6Only(conf.get<bool>("ipv6_only"))
            .dnsCacheTTL(pgg::Seconds{conf.get<unsigned long>("dns.cache_ttl")})
            .product();
    }

    void InitSharpei(const yplatform::ptree& conf) {
        SharpeiSettings.sharpeiAddress = {conf.get<std::string>("host"), conf.get<unsigned>("port")};
        SharpeiSettings.retries = conf.get<unsigned>("attempts");
    }

    void InitUserJournal(const yplatform::ptree& conf) {
        auto logger = std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, "userjournal");
        user_journal::ServiceFactory factory;
        factory.fileWriter([logger](const std::string &str) {
            YLOG((*logger), info) << str;
        });
        factory.tskvFormat("mail-user-journal-tskv-log");
        factory.tableName(conf.get<std::string>("table"));
        std::locale tskv_locale(NUtil::GetDefaultUtfLocale(), "C", std::locale::numeric);
        factory.locale(tskv_locale);
        UserJournalService = factory.product();
    }

    void InitSpamReportLogger(const yplatform::ptree& conf) {
        if (!conf.get("use_spam_reports", false)) {
            return;
        }
        SpamReportLogger = std::make_shared<yplatform::log::tskv_logger>(YGLOBAL_LOG_SERVICE, "spam_reports");
    }

    void LogSpamReport(
        const std::string& sessionId,
        const TUserInfo& userInfo,
        const TFilterActions& filterActions,
        const TStoreResult& res) const
    {
        if (res.SpamReport.OriginalFolder.Fid == res.SpamReport.DestinationFolder.Fid &&
            filterActions.AddedLids.empty() &&
            filterActions.AddedSymbols.empty())
        {
            return;
        }

        bool seen = AnyOf(res.SpamReport.AddedLabels, [](const auto& label) { return label.Symbol == "seen_label"; });
        namespace NTL = yplatform::log::typed;
        NTL::attributes_map common;
        common << NTL::make_attr("session", sessionId)
            << NTL::make_attr("uid", userInfo.Uid)
            << NTL::make_attr("suid", userInfo.Suid)
            << NTL::make_attr("ip", userInfo.IpFrom)
            << NTL::make_attr("folder", res.SpamReport.OriginalFolder.Name)
            << NTL::make_attr("dest_folder", res.SpamReport.DestinationFolder.Name)
            << NTL::make_attr("seen", seen ? "yes" : "no")
            << NTL::make_attr("mid", res.Envelope.mid())
            << NTL::make_attr("stid", res.Envelope.stid());

        auto now = TInstant::Now();
        common << NTL::make_attr("received_date", now.TimeT())
            << NTL::make_attr("timestamp", now.FormatLocalTime("%Y-%m-%d %H:%M:%S"))
            << NTL::make_attr("timezone", now.FormatLocalTime("%z"));

        if (res.SpamReport.OriginalFolder.Fid != res.SpamReport.DestinationFolder.Fid) {
            std::string type{"move"};
            if (res.SpamReport.DestinationFolder.TypeCode == macs::Folder::Symbol::trash.code()) {
                type = "delete";
            }
            YLOG(*SpamReportLogger, info) << common
                << NTL::make_attr("type", type);
        }

        for (const auto& label: res.SpamReport.AddedLabels) {
            YLOG(*SpamReportLogger, info) << common
                << NTL::make_attr("type", "flag:" + label.Name);
        }
    }

    sharpei::client::Settings SharpeiSettings;
    macs::pg::Credentials Credentials;
    macs::pg::QueryConf QueryConf;
    macs::pg::ConnectionPoolPtr ConnectionPool;
    macs::pg::Milliseconds TransactionTimeout;
    user_journal::ServicePtr UserJournalService;
    TSpamReportLogger SpamReportLogger;
};

} // namespace NMdb
