#pragma once

#include "context.h"
#include "logger.h"
#include "types.h"
#include "errors.h"
#include "mdb_module.h"

#include <macs/folder.h>
#include <macs/folder_set.h>
#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <boost/algorithm/string.hpp>

#include <vector>
#include <algorithm>

#include <util/generic/yexception.h>

namespace NMdb {

inline const macs::Folder::Symbol& GetFolderSymbolByName(const std::string& name) {
    Y_ENSURE(!name.empty());
    auto symbolicName = name.substr(1);
    boost::algorithm::to_lower(symbolicName);

    // Special hack for bug2bug compatibility with fastsrv
    return (symbolicName == "drafts" ? macs::Folder::Symbol::drafts : macs::Folder::Symbol::getByTitle(symbolicName));
}

struct TResolveOriginal {};
struct TResolveDestination {};

template <typename TFoldersRepository, typename TResolveType>
class TResolveFolderOp {
public:
    using THandler = std::function<void(boost::system::error_code, const TResolvedFolder&, TResolveType)>;
    using TYieldCtx = yplatform::yield_context<TResolveFolderOp<TFoldersRepository, TResolveType>>;

    TResolveFolderOp(
        TContextPtr context,
        TFoldersRepository foldersRepository,
        const TFolderCoords& coords,
        ENoSuchFolderAction noSuchFolderAction,
        TResolveType resolveType,
        const THandler& handler
    )
        : Context(context)
        , FoldersRepository(foldersRepository)
        , Coords(coords)
        , NoSuchFolderAction(noSuchFolderAction)
        , ResolveType(resolveType)
        , Handler(handler)
    {
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err = boost::system::error_code()) {
        try {
            reenter(yieldCtx) {
                yield FoldersRepository->getAllFoldersWithHidden(yieldCtx);
                if (err) {
                    yield break;
                }

                if (!Coords.Fid.empty()) {
                    err = ResolveFolderByFid();
                    yield break;
                } else {
                    yield ResolveFolderByPath(yieldCtx);
                }

                if (err) {
                    yield break;
                }

                yield FoldersRepository->getAllFoldersWithHidden(yieldCtx);
            }
        } catch (const std::exception& e) {
            MDBSAVE_LOG_ERROR(Context, logdog::exception=e, logdog::where_name="resolve_folders_op");
            return Handler(EError::OperationException, TResolvedFolder(), ResolveType);
        }

        if (yieldCtx.is_complete()) {
            if (err) {
                Handler(err, TResolvedFolder(), ResolveType);
            } else {
                auto resolvedFolder = BuildResolvedFolder();
                Handler(err, resolvedFolder, ResolveType);
            }
        }
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::FolderSet res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_folder_op", log::error_type="macs_error");
        } else {
            FolderSet = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::Folder macsFolder) {
        if (err == macs::error::folderCantBeParent) {
            if (Folder.fid().empty()) {
                MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_folder_op", log::error_type="macs_error");
                return (*this)(yieldCtx, EError::NoSuchFolderPerm);
            }
            MDBSAVE_LOG_NOTICE(Context,
                logdog::message="folder " + Folder.name() + " can't be parent, so use last resolved folder as a destination folder",
                logdog::where_name="resolve_folder_op"
            );
            return (*this)(yieldCtx, boost::system::error_code());
        } else if (err == macs::error::foldersLimitExceeded) {
            MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_folder_op", log::error_type="macs_error");
            return (*this)(yieldCtx, EError::NoSuchFolderPerm);
        } else if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_folder_op", log::error_type="macs_error");
        } else {
            Folder = macsFolder;
        }

        (*this)(yieldCtx, err.base());
    }

private:
    boost::system::error_code ResolveFolderByFid() {
        if (FolderSet.exists(Coords.Fid)) {
            Folder = FolderSet.at(Coords.Fid);
            return {};
        }

        if (NoSuchFolderAction == ENoSuchFolderAction::FallbackToInbox &&
            FolderSet.exists(macs::Folder::Symbol::inbox))
        {
            Folder = FolderSet.at(macs::Folder::Symbol::inbox);
            MDBSAVE_LOG_DEBUG(Context,
                logdog::message="no such folder, fid " + Coords.Fid + ", fallback to \\Inbox (fid: " + Folder.fid() + ")",
                logdog::where_name="resolve_folder_op");
            return {};
        }

        if (NoSuchFolderAction == ENoSuchFolderAction::FallbackToInbox) {
            MDBSAVE_LOG_ERROR(Context,
                logdog::message="no such folder, neither fid " + Coords.Fid + " nor symbol: \\Inbox exist",
                logdog::where_name="resolve_folder_op"
            );
        } else {
            MDBSAVE_LOG_ERROR(Context,
                logdog::message="no such folder, neither fid " + Coords.Fid,
                logdog::where_name="resolve_folder_op"
            );
        }
        return EError::NoSuchFolderPerm;
    }

    void ResolveFolderByPath(TYieldCtx yieldCtx) {
        auto [err, firstFolderSymbol, firstFolderName] = ResolveFirstFolderSymbol();
        if (err && err != EError::NoSuchFolderTemp) {
            return (*this)(yieldCtx, err);
        } else if (!firstFolderName.empty()) {
            Coords.Path[0] = firstFolderName;
        }

        if (firstFolderSymbol == macs::Folder::Symbol::none) {
            err = ResolvePath();
            if (err != EError::NoSuchFolderTemp) {
                return (*this)(yieldCtx, err);
            }
        }

        if (NoSuchFolderAction == ENoSuchFolderAction::Create) {
            return CreateFolderByPath(firstFolderSymbol, yieldCtx);
        }
        if (NoSuchFolderAction == ENoSuchFolderAction::FallbackToInbox) {
            MDBSAVE_LOG_NOTICE(Context,
                logdog::message="folder " + boost::join(Coords.Path, "/") + " not found, fallback to \\inbox",
                logdog::where_name="resolve_folder_op"
            );
            if (!FolderSet.exists(macs::Folder::Symbol::inbox)) {
                MDBSAVE_LOG_ERROR(Context, logdog::message="no \\inbox folder!", logdog::where_name="resolve_folder_op");
                err = EError::NoSuchFolderPerm;
            } else {
                Folder = FolderSet.at(macs::Folder::Symbol::inbox);
                err = EError::Ok;
            }
        } else {
            MDBSAVE_LOG_ERROR(Context,
                logdog::message="folder " + boost::join(Coords.Path, "/") + " not found!",
                logdog::where_name="resolve_folder_op"
            );
            err = EError::NoSuchFolderPerm;
        }
        return (*this)(yieldCtx, err);
    }

    TResolvedFolder BuildResolvedFolder() const {
        TResolvedFolder ret;
        ret.Fid = Folder.fid();
        ret.Name = FolderSet.getPath(Folder).toString();
        ret.Type = Folder.symbolicName().title();
        ret.TypeCode = Folder.symbolicName().code();
        return ret;
    }

    std::tuple<boost::system::error_code, macs::Folder::Symbol, std::string> ResolveFirstFolderSymbol() const {
        if (Coords.Path.empty() || Coords.Path[0].empty()) {
            MDBSAVE_LOG_ERROR(Context,
                logdog::message="empty folder path",
                logdog::where_name="resolve_folder_op"
            );
            return {EError::NoSuchFolderPerm, macs::Folder::Symbol::none, {}};
        }

        if (Coords.Path[0][0] != '\\') {
            return {{}, macs::Folder::Symbol::none, {}};
        }

        const auto symbol = GetFolderSymbolByName(Coords.Path[0]);
        if (symbol == macs::Folder::Symbol::none) {
            return {{}, symbol, {}};
        }

        if (!FolderSet.exists(symbol)) {
            if (macs::Folder::Symbol::isChangeble(symbol)) {
                MDBSAVE_LOG_DEBUG(Context,
                    logdog::message="no such folder, changeable symbol " + Coords.Path[0],
                    logdog::where_name="resolve_folder_op"
                );
                return {EError::NoSuchFolderTemp, symbol, {}};
            } else if (symbol == macs::Folder::Symbol::pending) {
                MDBSAVE_LOG_DEBUG(Context,
                    logdog::message="no such folder, symbol " + Coords.Path[0],
                    logdog::where_name="resolve_folder_op"
                );
                return {EError::NoSuchFolderTemp, symbol, {}};
            } else {
                MDBSAVE_LOG_ERROR(Context,
                    logdog::message="no such folder, symbol " + Coords.Path[0],
                    logdog::where_name="resolve_folder_op"
                );
                return {EError::NoSuchFolderPerm, macs::Folder::Symbol::none, {}};
            }
        }
        return {{}, macs::Folder::Symbol::none, FolderSet.at(symbol).name()};
    }

    void CreateFolderByPath(const macs::Folder::Symbol& symbolToCreate, TYieldCtx yieldCtx) {
        if ((symbolToCreate != macs::Folder::Symbol::none) && !FolderSet.exists(symbolToCreate)) {
            std::string name = symbolToCreate.title();
            if ((symbolToCreate == macs::Folder::Symbol::pending) && FolderSet.exists(name, macs::Folder::noParent)) {
                boost::uuids::random_generator generator;
                name += "_" + boost::uuids::to_string(generator());
            }

            return FoldersRepository->createFolder(name, macs::Folder::noParent, symbolToCreate,
                [this, yieldCtx](mail_errors::error_code err, macs::Folder macsFolder) mutable {
                    if (err) {
                        return (*this)(yieldCtx, err, macsFolder);
                    }

                    Coords.Path[0] = macsFolder.name();
                    return FoldersRepository->createFolderByPath(macs::Folder::Path(Coords.Path), yieldCtx);
                }
            );
        }

        return FoldersRepository->createFolderByPath(macs::Folder::Path(Coords.Path), yieldCtx);
    }

    boost::system::error_code ResolvePath() {
        auto parentFid = macs::Folder::noParent;
        for (const auto& name: Coords.Path) {
            if (!FolderSet.exists(name, parentFid)) {
                MDBSAVE_LOG_DEBUG(Context,
                    logdog::message="no such folder, name: " + name + ", parent_fid: " + parentFid,
                    logdog::where_name="resolve_folder_op"
                );
                return EError::NoSuchFolderTemp;
            }
            Folder = FolderSet.at(name, parentFid);
            parentFid = Folder.fid();
        }
        return boost::system::error_code();
    }

    TContextPtr Context;
    TFoldersRepository FoldersRepository;
    TFolderCoords Coords;
    ENoSuchFolderAction NoSuchFolderAction;
    TResolveType ResolveType;
    THandler Handler;
    macs::FolderSet FolderSet;
    macs::Folder Folder;
};

} // namespace NMdb

#include <yplatform/unyield.h>
