#pragma once

#include "context.h"
#include "logger.h"
#include "types.h"
#include "errors.h"
#include "mdb_module.h"

#include <macs/label.h>
#include <macs/label_set.h>
#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <util/generic/algorithm.h>

#include <vector>
#include <algorithm>

namespace NMdb {

template <typename TLabelsRepository>
class TResolveLabelsOp {
public:
    using THandler = std::function<void(boost::system::error_code, const std::vector<TResolvedLabel>&)>;
    using TYieldCtx = yplatform::yield_context<TResolveLabelsOp<TLabelsRepository>>;

    TResolveLabelsOp(
        TContextPtr context,
        TLabelsRepository labelsRepository,
        const std::vector<TLid>& lids,
        const std::vector<TLabelSymbol>& symbols,
        const std::vector<TLabel>& labels,
        const TResolvedFolder& folder,
        const THandler& handler
    )
        : Context(context)
        , LabelsRepository(labelsRepository)
        , Lids(lids)
        , Symbols(symbols)
        , Labels(labels)
        , Handler(handler)
    {
        if (folder.TypeCode == macs::Folder::Symbol::spam.code()) {
            this->Symbols.emplace_back("spam_label");
        }
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err = boost::system::error_code()) {
        try {
            reenter(yieldCtx) {
                yield LabelsRepository->getAllLabels(yieldCtx);
                if (err) {
                    yield break;
                }

                ExtractLabelsFromLids();

                for (SymbolIt = Symbols.begin(); SymbolIt != Symbols.end(); ++SymbolIt) {
                    LabelSymbol = macs::Label::Symbol::getByTitle(*SymbolIt);
                    if (LabelSymbol == macs::Label::Symbol::defValue()) {
                        MDBSAVE_LOG_WARN(Context,
                            logdog::message="symbol " + *SymbolIt + " not found",
                            logdog::where_name="resolve_labels_op"
                        );
                    } else {
                        yield LabelsRepository->getOrCreateLabel(LabelSymbol, yieldCtx);
                        if (err) {
                            // ignore error
                            err = EError::Ok;
                        }
                    }
                }

                for (LabelIt = Labels.begin(); LabelIt != Labels.end(); ++LabelIt) {
                    LabelType = macs::Label::Type::getByTitle(LabelIt->Type);

                    // first check label in labelSet because fake labels (FAKE_SEEN_LABEL,
                    // FAKE_SPAM_LABEL, etc) not resolved by getOrCreateLabel(name, type)
                    if (LabelSet.exists(LabelIt->Name, LabelType)) {
                        MacsLabels.push_back(LabelSet.at(LabelIt->Name, LabelType));
                    } else {
                        yield LabelsRepository->getOrCreateLabel(LabelIt->Name, LabelIt->Color, LabelType, yieldCtx);
                        if (err) {
                            // ignore error
                            err = EError::Ok;
                        }
                    }
                }

                MarkSeenIfMuted();

                Ret = GetUniqueFromLabels();
            }
        } catch (const std::exception& e) {
            MDBSAVE_LOG_ERROR(Context, logdog::exception=e, logdog::where_name="resolve_labels_op");
            err = EError::OperationException;
            return Handler(err, Ret);
        }

        if (yieldCtx.is_complete()) {
            Handler(err, Ret);
        }
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::LabelSet res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_labels_op", log::error_type="macs_error");
        } else {
            LabelSet = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::Label res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_labels_op", log::error_type="macs_error");
        } else {
            MacsLabels.push_back(std::move(res));
        }
        (*this)(yieldCtx, err.base());
    }

private:
    void ExtractLabelsFromLids() {
        for (const auto& lid: Lids) {
            auto it = LabelSet.find(lid);
            if (it != LabelSet.end()) {
                MacsLabels.push_back(it->second);
            } else {
                MDBSAVE_LOG_WARN(Context, logdog::message="lid " + lid + " not found", logdog::where_name="resolve_labels_op");
            }
        }
    }

    void MarkSeenIfMuted() {
        using Symbol = macs::Label::Symbol;
        if (!LabelSet.exists(Symbol::seen_label) ||
            !LabelSet.exists(Symbol::mute_label)) {
            return;
        }
        auto it = std::find_if(MacsLabels.begin(), MacsLabels.end(), [] (const macs::Label& label) {
            return label.symbolicName() == Symbol::mute_label;
        });
        if (it != MacsLabels.end()) {
            MacsLabels.emplace_back(LabelSet.at(Symbol::seen_label));
        }
    }

    std::vector<TResolvedLabel> GetUniqueFromLabels() const {
        std::vector<TResolvedLabel> labels;
        for (const auto& label: MacsLabels) {
            labels.push_back(TResolvedLabel{label.lid(), label.symbolicName().title(), label.name()});
        }

        SortUniqueBy(labels, [](const auto& label) { return label.Lid; });
        return labels;
    }

    TContextPtr Context;
    TLabelsRepository LabelsRepository;
    std::vector<TLid> Lids;
    std::vector<TLabelSymbol> Symbols;
    std::vector<TLabel> Labels;
    THandler Handler;
    macs::LabelSet LabelSet;
    macs::Label::Symbol LabelSymbol = macs::Label::Symbol::defValue();
    macs::Label::Type LabelType = macs::Label::Type::defValue();
    std::vector<TLabelSymbol>::const_iterator SymbolIt;
    std::vector<TLabel>::const_iterator LabelIt;
    std::vector<macs::Label> MacsLabels;
    std::vector<TResolvedLabel> Ret;
};

} // namespace NMdb

#include <yplatform/unyield.h>
