#pragma once

#include "context.h"
#include "logger.h"
#include "types.h"
#include "errors.h"
#include "mdb_module.h"

#include <macs/tab.h>
#include <macs/tab_set.h>
#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace NMdb {

template <typename TTabsRepository>
class TResolveTabOp {
public:
    using THandler = std::function<void(boost::system::error_code, const std::optional<macs::Tab::Type>&)>;
    using TYieldCtx = yplatform::yield_context<TResolveTabOp<TTabsRepository>>;

    TResolveTabOp(
        TContextPtr context,
        TTabsRepository tabsRepository,
        const TResolvedFolder& folder,
        const TMaybeFail<TTab>& tab,
        const THandler& handler
    )
        : Context(context)
        , TabsRepository(tabsRepository)
        , Folder(folder)
        , TabName(tab)
        , Handler(handler)
    {
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err = boost::system::error_code()) {
        try {
            reenter(yieldCtx) {
                if (Folder.TypeCode == macs::Folder::Symbol::inbox.code()) {
                    if (TabName) {
                        yield TabsRepository->getAllTabs(yieldCtx);
                        if (err) {
                            yield break;
                        }
                    }

                    if (!Ret) {
                        Ret = macs::Tab::Type::relevant;
                    }
                }
            }
        } catch (const std::exception& e) {
            MDBSAVE_LOG_ERROR(Context, logdog::exception=e, logdog::where_name="resolve_tab_op");
            err = EError::OperationException;
            return Handler(err, Ret);
        }

        if (yieldCtx.is_complete()) {
            Handler(err, Ret);
        }
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::TabSet res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::error_code=err, logdog::where_name="resolve_tab_op", log::error_type="macs_error");
        } else {
            if (res.exists(TabName.GetRef())) {
                Ret = res.at(TabName.GetRef()).type();
            }
        }
        (*this)(yieldCtx, err.base());
    }

private:
    TContextPtr Context;
    TTabsRepository TabsRepository;
    TResolvedFolder Folder;
    TMaybeFail<TTab> TabName;
    THandler Handler;
    std::optional<macs::Tab::Type> Ret;
};

} // namespace NMdb

#include <yplatform/unyield.h>
