#pragma once

#include <sharpei_client/sharpei_client.h>
#include <ymod_httpclient/client.h>
#include <yplatform/find.h>
#include <yplatform/util/sstream.h>
#include <boost/algorithm/string/join.hpp>

namespace NMdb {

namespace NSharpei = sharpei::client::http;

class TSharpeiHttpClient: public NSharpei::HttpClient {
public:
    using TAddress = NSharpei::Address;
    using TArguments = NSharpei::Arguments;
    using THeaders = NSharpei::Headers;
    using TTimeout = NSharpei::Timeout;
    using TResponse = NSharpei::Response;
    using THandler = NSharpei::ResponseHandler;

    explicit TSharpeiHttpClient(yplatform::task_context_ptr context)
        : Context(context)
    {
        HttpClient = yplatform::find<yhttp::call>("sharpei_client");
    }

    void aget(
        const TAddress& addr,
        TTimeout,
        const std::string& method,
        const TArguments& args,
        const THeaders& headers,
        THandler handler,
        bool,
        const std::string&) const override
    {
        DoRequest(yhttp::request::GET(MakeUrl(addr, method, args), MakeHeaders(headers)), handler);
    }

    void apost(
        const TAddress& addr,
        TTimeout,
        const std::string& method,
        const TArguments& args,
        const THeaders& headers,
        const std::string& data,
        THandler handler,
        bool,
        const std::string&) const override
    {
        DoRequest(yhttp::request::POST(MakeUrl(addr, method, args), MakeHeaders(headers), std::string(data)), handler);
    }

private:
    std::string MakeUrl(
        const TAddress& addr,
        const std::string& method,
        const TArguments& args) const
    {
        std::string url = addr.host;
        url += ":" + boost::lexical_cast<std::string>(addr.port);
        url += method;
        int par_n = 0;
        for (auto i: args) {
            for (auto j: i.second) {
                url += par_n++ == 0 ? "?" : "&";
                url += i.first;
                url += "=" + j;
            }
        }

        return url;
    }

    std::string MakeHeaders(const THeaders& headers) const {
        std::string result;
        yplatform::sstream str(result);
        for (const auto& header: headers) {
            str
                << header.first
                << ": "
                << boost::join(header.second, ", ")
                << "\r\n";
        }
        return result;
    }


    void DoRequest(yhttp::request req, THandler handler) const {
        HttpClient->async_run(Context, req,
            [handler](boost::system::error_code ec, yhttp::response resp) {
                handler(ec, TResponse{static_cast<unsigned>(resp.status), resp.body});
            });
    }


    yplatform::task_context_ptr Context;
    boost::shared_ptr<yhttp::call> HttpClient;
};

} // namespace NMdb
