#pragma once

#include "context.h"
#include "logger.h"
#include "resolve_folder_op.h"
#include "resolve_labels_op.h"
#include "resolve_tab_op.h"
#include "types.h"
#include "build_envelope.h"
#include "errors.h"
#include "update_store_result_with_resolved_labels.h"

#include <macs_pg/service/service.h>
#include <yplatform/task_context.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace NMdb {

class TStoreMailishOp {
public:
    using THandler = std::function<void(boost::system::error_code, const TStoreResult&)>;
    using TYieldCtx = yplatform::yield_context<TStoreMailishOp>;

    TStoreMailishOp(
        TContextPtr context,
        macs::ServicePtr db,
        const TMessageBase& message,
        const std::string& fid,
        const TLabelsArgs& labels,
        const TMaybeFail<TTab>& tab,
        const THeaders& headers,
        const std::vector<TAttachment>& attachments,
        const std::vector<TMimePart>& parts,
        const TThreadInfo& threadInfo,
        const TStoreParams& params,
        uint64_t externalImapId,
        const THandler& handler
    )
        : Context(context)
        , Db(db)
        , Message(message)
        , Fid(fid)
        , Headers(headers)
        , Attachments(attachments)
        , Labels(labels)
        , Tab(tab)
        , Params(params)
        , Handler(handler)
    {
        MimeParts = BuildMimeParts(parts);
        ThreadMeta = BuildThreadMeta(threadInfo);
        MailishInfo = macs::MailishMessageInfo{externalImapId, message.ReceivedDate};
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err = boost::system::error_code()) {
        try {
            reenter(yieldCtx) {
                yield ResolveFolder(yieldCtx);
                if (err) {
                    yield break;
                }

                yield ResolveLabels(yieldCtx);
                if (err) {
                    yield break;
                }

                yield ResolveTab(yieldCtx);
                if (err) {
                    yield break;
                }

                Envelope = BuildEnvelope(Message, Ret.Folder.Fid, Lids, ResolvedTab, Headers, Attachments, Context);
                Stid = Envelope.stid();

                yield Db->mailish().store(
                    std::move(Envelope),
                    std::move(MimeParts),
                    std::move(ThreadMeta),
                    Params.DisablePush
                        ? macs::EnvelopesRepository::NotificationMode::off
                        : macs::EnvelopesRepository::NotificationMode::on,
                    MailishInfo,
                    yieldCtx);

                if (err) {
                    err = EError::StoreToDbError;
                    yield break;
                }

                EntryKind = std::get<macs::EnvelopeKind>(SaveResult);
                if (EntryKind == macs::EnvelopeKind::duplicate) {
                    MDBSAVE_LOG_NOTICE(Context, logdog::message="duplicate found", logdog::where_name="store_mailish_op");
                    if (Params.NeedRemoveDuplicates) {
                        MDBSAVE_LOG_DEBUG(Context,
                            logdog::message="remove message from storage  by stid=" + Stid,
                            logdog::where_name="store_mailish_op"
                        );
                        yield Db->envelopes().deleteFromStorage(Stid, yieldCtx);
                        if (err) {
                            // just log
                            MDBSAVE_LOG_ERROR(Context,
                                logdog::message="failed to remove message from storage, stid=" + Stid,
                                logdog::error_code=err,
                                logdog::where_name="store_mailish_op"
                            );
                        }
                    }
                }

                Ret.Envelope = std::get<macs::Envelope>(SaveResult);
                Ret.Duplicate = (EntryKind == macs::EnvelopeKind::duplicate);

                yield UpdateStoreResultWithResolvedLabels(Db, Ret, yieldCtx);
            }
        } catch (const std::exception& e) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="store_mailish_op", logdog::exception=e);
            err = EError::OperationException;
        }

        if (yieldCtx.is_complete()) {
            Handler(err, Ret);
        }
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::UpdateEnvelopeResult res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="store_mailish_op", logdog::error_code=err, log::error_type="macs_error");
        } else {
            SaveResult = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="store_mailish_op", logdog::error_code=err, log::error_type="macs_error");
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err, const std::vector<TResolvedLabel>& res) {
        if (!err) {
            for (const auto& label: res) {
                Lids.emplace_back(label.Lid);
            }
            Ret.Labels = res;
        }
        (*this)(yieldCtx, err);
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err, const TResolvedFolder& res, TResolveDestination) {
        if (!err) {
            Ret.Folder = res;
        }
        (*this)(yieldCtx, err);
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::LabelSet res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="store_mailish_op", logdog::error_code=err, log::error_type="macs_error");
        } else {
            AllLabels = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err, const std::optional<macs::Tab::Type>& res) {
        if (!err) {
            ResolvedTab = res;
        }

        (*this)(yieldCtx, err);
    }

private:
    void ResolveLabels(TYieldCtx yieldCtx) {
        using TResolveOp = TResolveLabelsOp<macs::LabelsRepository*>;
        auto resolveLabelsOp = std::make_shared<TResolveOp>(
            Context,
            &Db->labels(),
            Labels.Lids,
            Labels.Symbols,
            Labels.Labels,
            Ret.Folder,
            yieldCtx);
        yplatform::spawn(resolveLabelsOp);
    }

    void ResolveFolder(TYieldCtx yieldCtx) {
        using TResolveOp = TResolveFolderOp<macs::FoldersRepository*, TResolveDestination>;
        TFolderCoords destFolder;
        destFolder.Fid = Fid;
        auto resolveFolderOp = std::make_shared<TResolveOp>(
            Context,
            &Db->folders(),
            destFolder,
            ENoSuchFolderAction::Fail,
            TResolveDestination{},
            yieldCtx);
        yplatform::spawn(resolveFolderOp);
    }

    void ResolveTab(TYieldCtx yieldCtx) {
        using TResolveOp = TResolveTabOp<const macs::TabsRepository*>;
        auto resolveTabOp = std::make_shared<TResolveOp>(Context, &Db->tabs(), Ret.Folder, Tab, yieldCtx);
        yplatform::spawn(resolveTabOp);
    }

    TContextPtr Context;
    std::vector<TLid> Lids;
    std::string Stid;
    macs::UpdateEnvelopeResult SaveResult;
    macs::EnvelopeKind EntryKind;
    TStoreResult Ret;
    macs::ServicePtr Db;
    macs::MimeParts MimeParts;
    macs::ThreadMeta ThreadMeta;
    macs::MailishMessageInfo MailishInfo;
    macs::Envelope Envelope;
    macs::LabelSet AllLabels;
    TMessageBase Message;
    std::string Fid;
    THeaders Headers;
    std::vector<TAttachment> Attachments;
    TLabelsArgs Labels;
    TMaybeFail<TTab> Tab;
    TStoreParams Params;
    std::optional<macs::Tab::Type> ResolvedTab;
    THandler Handler;
};

} // namespace NMdb

#include <yplatform/unyield.h>
