#pragma once

#include "context.h"
#include "logger.h"
#include "resolve_folder_op.h"
#include "resolve_labels_op.h"
#include "types.h"
#include "build_envelope.h"
#include "errors.h"
#include "update_store_result_with_resolved_labels.h"

#include <macs_pg/service/service.h>
#include <yplatform/task_context.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace NMdb {

class TUpdateEnvelopeOp {
public:
    using THandler = std::function<void(boost::system::error_code, const TStoreResult&)>;
    using TYieldCtx = yplatform::yield_context<TUpdateEnvelopeOp>;

    TUpdateEnvelopeOp(
        TContextPtr context,
        macs::ServicePtr db,
        const std::string& oldMid,
        const TMessageBase& message,
        const std::string& fid,
        const TLabelsArgs& labels,
        const THeaders& headers,
        const std::vector<TAttachment>& attachments,
        const std::vector<TMimePart>& parts,
        const TThreadInfo& threadInfo,
        const TStoreParams& params,
        const THandler& handler
    )
        : Context(context)
        , Db(db)
        , OldMid(oldMid)
        , Message(message)
        , Fid(fid)
        , Headers(headers)
        , Attachments(attachments)
        , Labels(labels)
        , Params(params)
        , Handler(handler)
    {
        MimeParts = BuildMimeParts(parts);
        ThreadMeta = BuildThreadMeta(threadInfo);
        SaveOpts.ignoreDuplicates = Params.IgnoreDuplicates;
        SaveOpts.notificationMode = Params.DisablePush
            ? macs::EnvelopesRepository::NotificationMode::off
            : macs::EnvelopesRepository::NotificationMode::on;
    }

    void operator() (TYieldCtx yieldCtx, boost::system::error_code err = boost::system::error_code()) {
        try {
            reenter(yieldCtx) {
                yield ResolveFolder(yieldCtx);
                if (err) {
                    yield break;
                }

                yield ResolveLabels(yieldCtx);
                if (err) {
                    yield break;
                }

                Envelope = BuildEnvelope(Message, Ret.Folder.Fid, Lids, std::nullopt, Headers, Attachments, Context);

                yield Db->labels().getAllLabels(yieldCtx);
                if (err) {
                    yield break;
                }

                DraftIt = AllLabels.find(macs::Label::Symbol::draft_label);
                DelayedIt = AllLabels.find(macs::Label::Symbol::delayedMessage_label);

                if (!OldMid.empty() &&
                    ((DraftIt != AllLabels.end() && Envelope.hasLabel(DraftIt->second.lid())) ||
                        (DelayedIt != AllLabels.end() && Envelope.hasLabel(DelayedIt->second.lid()))))
                {
                    MDBSAVE_LOG_DEBUG(Context, logdog::message="get message", log::mid=OldMid, logdog::where_name="update_envelope_op");
                    yield Db->envelopes().getById(OldMid, yieldCtx);
                    if (err) {
                        // ignore error
                    }
                }

                Stid = Envelope.stid();

                if (Old.mid().empty()) {
                    yield Db->envelopes().save(
                        std::move(Envelope),
                        std::move(MimeParts),
                        [this](const macs::Envelope&) { return ThreadMeta; },
                        std::move(SaveOpts),
                        yieldCtx);
                } else {
                    yield Db->envelopes().update(
                        Old,
                        macs::EnvelopeFactory(std::move(Envelope)).threadId(Old.threadId()).release(),
                        std::move(MimeParts),
                        [this](const macs::Envelope&) { return ThreadMeta; },
                        SaveOpts.ignoreDuplicates,
                        yieldCtx);
                }

                if (err) {
                    err = EError::StoreToDbError;
                    yield break;
                }

                Envelope = std::get<macs::Envelope>(SaveResult);
                EntryKind = std::get<macs::EnvelopeKind>(SaveResult);

                if (EntryKind == macs::EnvelopeKind::duplicate) {
                    MDBSAVE_LOG_NOTICE(Context, logdog::message="duplicate found", logdog::where_name="update_envelope_op");
                    if (Params.NeedRemoveDuplicates) {
                        yield RemoveFromStorage(yieldCtx, Stid);
                        if (err) {
                            // just log
                            MDBSAVE_LOG_ERROR(Context,
                                logdog::message="failed to remove message from storage, stid=" + Stid,
                                logdog::error_code=err,
                                logdog::where_name="update_envelope_op"
                            );
                        }
                    }
                } else {
                    if (!Old.mid().empty()) {
                        if (Envelope.mid() != Old.mid()) {
                            MDBSAVE_LOG_DEBUG(Context, logdog::message="remove message", log::mid=Old.mid(), logdog::where_name="update_envelope_op");
                            yield Db->envelopes().forceRemove(Old.mid(), yieldCtx);
                            if (err) {
                                // just log
                                MDBSAVE_LOG_ERROR(Context,
                                    logdog::message="failed to remove message",
                                    logdog::error_code=err,
                                    log::mid=Old.mid(),
                                    logdog::where_name="update_envelope_op"
                                );
                            }
                        } else {
                            yield RemoveFromStorage(yieldCtx, Old.stid());
                            if (err) {
                                // just log
                                MDBSAVE_LOG_ERROR(Context,
                                    logdog::message="failed to remove message from storage, stid=" + Old.stid(),
                                    logdog::error_code=err,
                                    logdog::where_name="update_envelope_op"
                                );
                            }
                        }
                    }
                }

                Ret.Envelope = Envelope;
                Ret.Duplicate = (EntryKind == macs::EnvelopeKind::duplicate);

                yield UpdateStoreResultWithResolvedLabels(Db, Ret, yieldCtx);
            }
        } catch (const std::exception& e) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="update_envelope_op", logdog::exception=e);
            err = EError::OperationException;
        }

        if (yieldCtx.is_complete()) {
            Handler(err, Ret);
        }
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::UpdateEnvelopeResult res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="update_envelope_op", logdog::error_code=err, log::error_type="macs_error");
        } else {
            SaveResult = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::Envelope res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="update_envelope_op", logdog::error_code=err, log::error_type="macs_error");
        } else {
            Old = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, macs::LabelSet res) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="update_envelope_op", logdog::error_code=err, log::error_type="macs_error");
        } else {
            AllLabels = std::move(res);
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err, const macs::Revision&) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="update_envelope_op", logdog::error_code=err, log::error_type="macs_error");
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, mail_errors::error_code err) {
        if (err) {
            MDBSAVE_LOG_ERROR(Context, logdog::where_name="update_envelope_op", logdog::error_code=err, log::error_type="macs_error");
        }
        (*this)(yieldCtx, err.base());
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err, const std::vector<TResolvedLabel>& res) {
        if (!err) {
            for (const auto& label: res) {
                Lids.emplace_back(label.Lid);
            }
            Ret.Labels = res;
        }
        (*this)(yieldCtx, err);
    }

    void operator()(TYieldCtx yieldCtx, boost::system::error_code err, const TResolvedFolder& res, TResolveDestination) {
        if (!err) {
            Ret.Folder = res;
        }
        (*this)(yieldCtx, err);
    }

private:
    void RemoveFromStorage(TYieldCtx yieldCtx, const std::string& stid) {
        MDBSAVE_LOG_DEBUG(Context, logdog::message="remove message from storage by stid=" + stid, logdog::where_name="update_envelope_op");
        Db->envelopes().deleteFromStorage(stid, yieldCtx);
    }

    void ResolveLabels(TYieldCtx yieldCtx) {
        using TResolveOp = TResolveLabelsOp<macs::LabelsRepository*>;
        auto resolveLabelsOp = std::make_shared<TResolveOp>(
            Context,
            &Db->labels(),
            Labels.Lids,
            Labels.Symbols,
            Labels.Labels,
            Ret.Folder,
            yieldCtx);
        yplatform::spawn(resolveLabelsOp);
    }

    void ResolveFolder(TYieldCtx yieldCtx) {
        using TResolveOp = TResolveFolderOp<macs::FoldersRepository*, TResolveDestination>;
        TFolderCoords destFolder;
        destFolder.Fid = Fid;
        auto resolveFolderOp = std::make_shared<TResolveOp>(
            Context,
            &Db->folders(),
            destFolder,
            ENoSuchFolderAction::Fail,
            TResolveDestination{},
            yieldCtx);
        yplatform::spawn(resolveFolderOp);
    }

    TContextPtr Context;
    std::vector<TLid> Lids;
    std::string Stid;
    macs::UpdateEnvelopeResult SaveResult;
    macs::EnvelopeKind EntryKind;
    TStoreResult Ret;
    macs::ServicePtr Db;
    std::string OldMid;
    macs::MimeParts MimeParts;
    macs::ThreadMeta ThreadMeta;
    macs::EnvelopesRepository::SaveOptions SaveOpts;
    macs::Envelope Envelope, Old;
    macs::LabelSet AllLabels;
    macs::LabelSet::const_iterator DelayedIt, DraftIt;
    TMessageBase Message;
    std::string Fid;
    THeaders Headers;
    std::vector<TAttachment> Attachments;
    TLabelsArgs Labels;
    TStoreParams Params;
    THandler Handler;
};

} // namespace NMdb

#include <yplatform/unyield.h>
