#pragma once

#include <mail/mdbsave/lib/mdb/context.h>
#include <mail/mdbsave/lib/mdb/logger.h>
#include <mail/mdbsave/lib/web/response.h>
#include <mail/mdbsave/lib/web/types/common.h>
#include <mail/mdbsave/lib/web/types/response.h>
#include <mail/mdbsave/lib/web/util/exception.h>
#include <mail/mdbsave/lib/web/util/get_header.h>

#include <functional>

namespace NMdb::NWeb::NHandler {

namespace NDetail {

template<typename THandler>
decltype(auto) Unwrap(THandler&& handler)
{
    return std::forward<THandler>(handler);
}

template<typename THandler>
THandler& Unwrap(std::shared_ptr<THandler>& handler)
{
    return *handler;
}

}

template <class TTvmGuardPtr, class THandlerPtr>
class TBase {
public:
    TBase(TTvmGuardPtr tvmGuard, std::string uri, EHttpMethod method, THandlerPtr handler)
        : TvmGuard(tvmGuard)
        , Uri(std::move(uri))
        , Method(method)
        , Handler(std::move(handler))
    {}

    template <class ...Ts>
    void operator() (THttpStreamPtr stream, std::string SessionId, Ts&&... args) {
        const auto req = stream->request();
        auto ctx = MakeContext(std::move(SessionId));

        if (req->method != Method) {
            const auto methodNotAllowed = EHttpReason::get(EHttpCode::method_not_allowed);
            MDBSAVE_LOG_ERROR(ctx, logdog::message=methodNotAllowed);
            return Respond(stream, EHttpCode::method_not_allowed, TErrorResponse{methodNotAllowed, methodNotAllowed});
        }

        const auto tvmResult = TvmGuard->check(
            Uri,
            {},
            GetHeaderOptional(req->headers, tvm_guard::header::serviceTicket()),
            GetHeaderOptional(req->headers, tvm_guard::header::userTicket())
        );

        if (tvmResult.action == tvm_guard::Action::reject) {
            MDBSAVE_LOG_ERROR(ctx, logdog::message=tvm_guard::toString(tvmResult.reason));
            return Respond(
                stream,
                EHttpCode::unauthorized,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::unauthorized),
                    tvm_guard::toString(tvmResult.reason)});
        }

        try {
            NDetail::Unwrap(Handler)(ctx, stream, std::forward<Ts>(args)...);
        } catch (const TInvalidArgument& ex) {
            MDBSAVE_LOG_ERROR(ctx, logdog::message="cannot parse request", logdog::exception=ex);
            Respond(
                stream,
                EHttpCode::bad_request,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::bad_request),
                    ex.what()});
        }  catch(const std::exception& ex) {
            MDBSAVE_LOG_ERROR(ctx, logdog::message="error occured", logdog::exception=ex);
            return Respond(
                stream,
                EHttpCode::internal_server_error,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::internal_server_error),
                    ex.what()});
        } catch(...) {
            MDBSAVE_LOG_ERROR(ctx, logdog::message="unknown error occured");
            return Respond(
                stream,
                EHttpCode::internal_server_error,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::internal_server_error),
                    "unknown error occured"});
        }
    }

    std::string GetUri() const {
        return Uri;
    }

    TContextPtr MakeContext(std::string sessionId) const {
        return boost::make_shared<TContext>(
            std::move(sessionId),
            yplatform::log::source {YGLOBAL_LOG_SERVICE, "mdbsave"});
    }

private:
    TTvmGuardPtr TvmGuard;
    std::string Uri;
    EHttpMethod Method;
    THandlerPtr Handler;
};

}
