#pragma once

#include <mail/mdbsave/lib/mdb/context.h>
#include <mail/mdbsave/lib/mdb/errors.h>
#include <mail/mdbsave/lib/mdb/mdb_module.h>
#include <mail/mdbsave/lib/mdb/logger.h>
#include <mail/mdbsave/lib/web/types/common.h>
#include <mail/mdbsave/lib/web/types/reflection/request.h>
#include <mail/mdbsave/lib/web/types/reflection/response.h>
#include <mail/mdbsave/lib/web/types/request.h>
#include <mail/mdbsave/lib/web/types/response.h>
#include <mail/mdbsave/lib/web/util/exception.h>
#include <mail/mdbsave/lib/web/util/helpers.h>

#include <yplatform/find.h>
#include <yplatform/coroutine.h>
#include <yplatform/module.h>
#include <yplatform/yield.h>

#include <yamail/data/deserialization/yajl.h>

namespace NMdb::NWeb::NHandler {

class TSaveOp {
public:
    using TYieldCtx = yplatform::yield_context<TSaveOp>;

    TSaveOp(TContextPtr ctx, THttpStreamPtr stream)
        : Ctx(std::move(ctx))
        , Stream(std::move(stream))
    {}

    void operator()(TYieldCtx yieldCtx) {
        try {
            reenter(yieldCtx) {
                Request = ParseRequest();

                if (Request.Sync && yplatform::exists<NMdb::TMdbModule>("mdb_sync")) {
                    Mdb = yplatform::find<NMdb::TMdbModule, std::shared_ptr>("mdb_sync");
                } else {
                    Mdb = yplatform::find<NMdb::TMdbModule, std::shared_ptr>("mdb");
                }

                for (RcptIt = Request.Rcpts.begin(); RcptIt != Request.Rcpts.end(); ++RcptIt) {
                    yield Save(yieldCtx.capture(Ec, StoreResult));
                    MakeResponseForRcpt();
                }
            }
        } catch (const TInvalidArgument& ex) {
            MDBSAVE_LOG_ERROR(Ctx, logdog::where_name=Where, logdog::message="cannot parse request", logdog::exception=ex);
            Respond(
                Stream,
                EHttpCode::bad_request,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::bad_request),
                    ex.what()});
        } catch (const std::exception& ex) {
            MDBSAVE_LOG_ERROR(Ctx, logdog::where_name=Where, logdog::message="error occured", logdog::exception=ex);
            return Respond(
                Stream,
                EHttpCode::internal_server_error,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::internal_server_error),
                    ex.what()});
        } catch (...) {
            MDBSAVE_LOG_ERROR(Ctx, logdog::where_name=Where, logdog::message="unknown error occured");
            return Respond(
                Stream,
                EHttpCode::internal_server_error,
                TErrorResponse{
                    EHttpReason::get(EHttpCode::internal_server_error),
                    "unknown error occured"});
        }

        if (yieldCtx.is_complete()) {
            Respond(Stream, EHttpCode::ok, Response);
        }
    }

private:
    TSaveRequest ParseRequest() const {
        const auto req = Stream->request();
        const std::string body(req->raw_body.cbegin(), req->raw_body.cend());

        try {
            return yamail::data::deserialization::fromJson<TSaveRequest>(body);
        } catch (const std::exception& ex) {
            throw TInvalidArgument{ex.what()};
        }
    };

    void Save(auto handler) {
        try {
            const auto& rcpt = RcptIt->Rcpt;
            const std::string oldMid = rcpt.Message.OldMid.value_or("");
            const std::int64_t externalImapId = rcpt.Message.ExtImapId.value_or(0);
            const auto userInfo = BuildUserInfo(rcpt.User);
            const auto messageInfo = BuildMessageInfo(rcpt.Message);
            const auto destFolder = BuildFolder(rcpt.Folders);
            const auto labels = BuildLabels(rcpt.Message.Lids, rcpt.Message.LabelSymbols, rcpt.Message.Labels, rcpt.AddedLids, rcpt.AddedSymbols);
            const auto tab = BuildTab(rcpt.Message.Tab);
            const auto headers = BuildHeaders(rcpt.Message.Headers);
            const auto& attachments = rcpt.Message.Attachments.value_or(std::vector<NMdb::TAttachment>());
            const auto mimeParts = BuildMimeParts(rcpt.Message.MimeParts, messageInfo.OffsetDiff);
            const auto threadInfo = BuildThreadInfo(rcpt.Message.ThreadInfo);
            const auto storeParams = BuildStoreParams(rcpt.Actions, rcpt.Imap);
            const auto noSuchFolderAction = BuildNoSuchFolderAction(rcpt.Actions);
            const auto filterActions = BuildFilterActions(rcpt.Actions, rcpt.Folders, rcpt.AddedLids, rcpt.AddedSymbols);

            if (!oldMid.empty()) {
                Mdb->Update(
                    Ctx,
                    Ctx->sessionId(),
                    userInfo,
                    oldMid,
                    messageInfo,
                    destFolder.Fid,
                    labels,
                    headers,
                    attachments,
                    mimeParts,
                    threadInfo,
                    storeParams,
                    handler);
            } else if (externalImapId) {
                Mdb->StoreMailish(
                    Ctx,
                    Ctx->sessionId(),
                    userInfo,
                    messageInfo,
                    destFolder.Fid,
                    labels,
                    tab,
                    headers,
                    attachments,
                    mimeParts,
                    threadInfo,
                    storeParams,
                    externalImapId,
                    handler);
            } else {
                Mdb->Save(
                    Ctx,
                    Ctx->sessionId(),
                    userInfo,
                    messageInfo,
                    destFolder,
                    labels,
                    tab,
                    headers,
                    attachments,
                    mimeParts,
                    threadInfo,
                    storeParams,
                    noSuchFolderAction,
                    filterActions,
                    handler);
            }
        } catch (const std::exception& e) {
            MDBSAVE_LOG_NOTICE(Ctx,
                logdog::exception=e,
                logdog::where_name="save_op",
                log::uid=RcptIt->Rcpt.User.Uid,
                logdog::message="save error, recipient_id=" + RcptIt->Id
            );
            handler(NMdb::EError::OperationException, NMdb::TStoreResult{});
        }
    }

    void MakeResponseForRcpt() {
        TSaveResponseRcptNode node;
        node.Id = RcptIt->Id;
        node.Rcpt.Uid = RcptIt->Rcpt.User.Uid;

        if (Ec) {
            MDBSAVE_LOG_ERROR(Ctx,
                logdog::message="save error, recipient_id=" + RcptIt->Id,
                logdog::error_code=Ec,
                log::uid=RcptIt->Rcpt.User.Uid
            );
            node.Rcpt.Status = (Ec == NMdb::EError::NoSuchFolderPerm ? "perm error" : "temp error");
            node.Rcpt.Description = Ec.message();
        } else {
            MDBSAVE_LOG_NOTICE(Ctx,
                logdog::message="save success, recipient_id=" + RcptIt->Id,
                log::imap_id=StoreResult.Envelope.imapId(),
                log::uid=RcptIt->Rcpt.User.Uid,
                log::duplicate=StoreResult.Duplicate,
                log::mid=StoreResult.Envelope.mid()
            );
            node.Rcpt.Status = "ok";
            node.Rcpt.Mid = StoreResult.Envelope.mid();
            node.Rcpt.ImapId = StoreResult.Envelope.imapId();
            node.Rcpt.Tid = StoreResult.Envelope.threadId();
            node.Rcpt.Duplicate = StoreResult.Duplicate;
            node.Rcpt.Folder = std::move(StoreResult.Folder);
            if (StoreResult.Labels.size()) {
                node.Rcpt.Labels = std::vector<TResponseLabel>();
                node.Rcpt.Labels->reserve(StoreResult.Labels.size());
                for (const auto& label: StoreResult.Labels) {
                    node.Rcpt.Labels->emplace_back(TResponseLabel {label.Lid, label.Symbol});
                }
            }
        }
        Response.Rcpts.emplace_back(std::move(node));
    }

    TContextPtr Ctx;
    THttpStreamPtr Stream;
    TSaveRequest Request;
    std::vector<TSaveRequestRcptNode>::iterator RcptIt;
    std::shared_ptr<NMdb::TMdbModule> Mdb;
    TSaveResponse Response;
    boost::system::error_code Ec;
    NMdb::TStoreResult StoreResult;
    const std::string Where {"save handler"};
};

#include <yplatform/unyield.h>

class TSave {
public:
    void operator() (TContextPtr ctx, THttpStreamPtr stream) {
        auto saveOp = std::make_shared<TSaveOp>(ctx, stream);
        yplatform::spawn(saveOp);
    }
};

}
