#pragma once

#include <mail/mdbsave/lib/mdb/types.h>
#include <mail/mdbsave/lib/mdb/mdb_module.h>
#include <mail/mdbsave/lib/web/types/request.h>

#include <optional>

namespace NMdb::NWeb {

inline NMdb::TUserInfo BuildUserInfo(const TUser& user) {
    NMdb::TUserInfo userInfo;
    userInfo.Uid = user.Uid;
    userInfo.Suid = user.Suid.value_or(userInfo.Suid);
    return userInfo;
}

inline NMdb::TMessageBase BuildMessageInfo(const TMessage& message) {
    NMdb::TMessageBase messageBase;
    messageBase.Stid = message.Storage.Stid;
    messageBase.OffsetDiff = message.Storage.Offset ? *message.Storage.Offset : 0;
    messageBase.FirstLine = message.Firstline;
    messageBase.ReceivedDate = message.Headers.RecievedDate;
    messageBase.Size = (message.Size ? *message.Size : 0) + messageBase.OffsetDiff;
    return messageBase;
}

inline std::optional<NMdb::TFolderCoords> BuildFolder(const TRequestFolder& folder) {
    std::optional<NMdb::TFolderCoords> folderCoords;
    if (folder.Fid) {
        folderCoords = NMdb::TFolderCoords();
        folderCoords->Fid = *folder.Fid;
        return folderCoords;
    }

    if (folder.Path && folder.Path->Path) {
        folderCoords = NMdb::TFolderCoords();
        if (folder.Path->Delimeter) {
            folderCoords->Path = yplatform::util::split(*folder.Path->Path, *folder.Path->Delimeter);
        } else {
            folderCoords->Path.emplace_back(*folder.Path->Path);
        }
    }
    return folderCoords;
}

inline NMdb::TFolderCoords BuildFolder(const TRequestFolders& folders) {
    std::optional<NMdb::TFolderCoords> folderCoords;
    if (folders.Destination) {
        folderCoords = BuildFolder(*folders.Destination);
    }

    if (!folderCoords && folders.Original) {
        folderCoords = BuildFolder(*folders.Original);
    }

    if (!folderCoords) {
        folderCoords = NMdb::TFolderCoords();
        folderCoords->Path.emplace_back("\\Inbox");
    }
    return *folderCoords;
}

inline NMdb::TLabelsArgs BuildLabels(
    const std::optional<TRequestLids>& lids,
    const std::optional<TRequestLabelSymbols>& labelSymbols,
    const std::optional<TRequestLabels>& labels,
    const std::optional<TRequestAddedLids>& addedLids,
    const std::optional<TRequestAddedSymbols>& addedSymbols)
{
    auto MakeOp = [] (const auto& source, auto& destination) {
        if (source) {
            for (const auto& val: *source) {
                destination.emplace_back(val);
            }
        }
    };

    NMdb::TLabelsArgs labelsArgs;
    MakeOp(lids, labelsArgs.Lids);
    MakeOp(addedLids, labelsArgs.Lids);
    MakeOp(labelSymbols, labelsArgs.Symbols);
    MakeOp(addedSymbols, labelsArgs.Symbols);

    if (labels) {
        for (const auto& label: *labels) {
            NMdb::TLabel item;
            item.Name = label.Name;
            item.Type = label.Type;
            if (item.Type == "user") {
                item.Color = "16711680";
            }
            labelsArgs.Labels.emplace_back(std::move(item));
        }
    }

    return labelsArgs;
}

inline TMaybeFail<NMdb::TTab> BuildTab(const std::optional<std::string>& tab) {
    return tab ? *tab : TMaybeFail<NMdb::TTab>{};
}

inline std::string BuildAddresses(const std::vector<THeaderAddress>& addresses) {
    std::string result;
    for (const auto& address: addresses) {
        if (!result.empty()) {
            result += ",";
        }
        const std::string name = address.DisplayName;
        if (!name.empty()) {
            result += "\"" + name + "\" ";
        }
        const std::string domain = address.Domain;
        const std::string local = address.Local;
        if (domain.empty()) {
            result += "<" + local + ">";
        } else {
            result += "<" + local + "@" + domain  + ">";
        }
    }
    return result;
}

inline std::string BuildToAndCc(const std::vector<THeaderAddress>& to, const std::optional<std::vector<THeaderAddress>>& cc) {
    std::string result;
    auto MakeAddress = [&result](const std::vector<THeaderAddress>& addresses) {
        for (const auto& address: addresses) {
            if (!result.empty()) {
                result += ",";
            }
            const std::string domain = address.Domain;
            result += address.Local;
            if (!domain.empty()) {
                result += "@" + domain;
            }
        }
    };
    MakeAddress(to);
    if (cc) {
        MakeAddress(*cc);
    }
    return result;
}

inline NMdb::THeaders BuildHeaders(const THeaders& headers) {
    NMdb::THeaders result;
    result.Subject = headers.Subject;
    result.MessageId = headers.MsgId.value_or(result.MessageId);
    result.Date = headers.Date.value_or(result.Date);
    result.ReplyTo = headers.ReplyTo.value_or(result.ReplyTo);
    result.InReplyTo = headers.InReplyTo.value_or(result.InReplyTo);

    result.To = BuildAddresses(headers.To);
    result.From = BuildAddresses(headers.From);

    if (headers.Cc) {
        result.Cc = BuildAddresses(*headers.Cc);

    }

    if (headers.Bcc) {
        result.Bcc = BuildAddresses(*headers.Bcc);
    }

    result.ToAndCc = BuildToAndCc(headers.To, headers.Cc);
    return result;
}

inline std::vector<NMdb::TMimePart> BuildMimeParts(
    const std::optional<std::vector<TMimePart>>& parts,
    std::size_t offsetDiff)
{
    std::vector<NMdb::TMimePart> result;
    if (parts) {
        for (const auto& part: *parts) {
            NMdb::TMimePart item;
            item.Hid = part.Hid;
            item.ContentType = part.ContentType;
            item.ContentSubtype = part.ContentSubtype;
            item.Boundary = part.Boundary.value_or(item.Boundary);
            item.Name = part.Name.value_or(item.Name);
            item.Charset = part.Charset.value_or(item.Charset);
            item.Encoding = part.Encoding.value_or(item.Encoding);
            item.ContentDisposition = part.ContentDisposition.value_or(item.ContentDisposition);
            item.FileName = part.FileName.value_or(item.FileName);
            item.ContentId = part.ContentId.value_or(item.ContentId);
            item.Length = part.Length.value_or(item.Length);
            item.Offset = part.Offset.value_or(0) + offsetDiff;
            result.emplace_back(std::move(item));
        }
    }
    return result;
}

inline NMdb::EThreadsHashNamespaces BuildThreadsHashNamespace(const std::string& ns) {
    if (ns == "subject") {
        return NMdb::EThreadsHashNamespaces::Subject;
    } else if (ns == "from") {
        return NMdb::EThreadsHashNamespaces::From;
    } else {
        throw std::runtime_error("unknown thread namespace \"" + ns + "\"");
    }
}

inline NMdb::EThreadsMergeRules BuildThreadsMergeRule(const std::string& rule) {
    if (rule == "force_new") {
        return NMdb::EThreadsMergeRules::ForceNewThread;
    } else if (rule == "hash") {
        return NMdb::EThreadsMergeRules::Hash;
    } else if (rule == "references") {
        return NMdb::EThreadsMergeRules::References;
    } else {
        throw std::runtime_error("unknown thread merge rule \"" + rule + "\"");
    }
}

inline NMdb::TThreadInfo BuildThreadInfo(const TThreadInfo& threadInfo) {
    NMdb::TThreadInfo result;

    result.Limits.DaysLimit = threadInfo.Limits.Days;
    result.Limits.CountLimit = threadInfo.Limits.Count;;
    result.Hash.Value = threadInfo.Hash.Value;
    result.Hash.Ns = BuildThreadsHashNamespace(threadInfo.Hash.Namespace);
    result.MergeRule = BuildThreadsMergeRule(threadInfo.Rule);
    result.ReferenceHashes = threadInfo.ReferenceHashes;
    result.MessageIds = threadInfo.MessageIds;
    result.InReplyToHash = threadInfo.InReplyToHash;
    result.MessageIdHash = threadInfo.MessageIdHash;
    return result;
}

inline NMdb::TStoreParams BuildStoreParams(const TActions& actions, bool Imap) {
    NMdb::TStoreParams result;
    result.IgnoreDuplicates = actions.Duplicates.Ignore;
    result.NeedRemoveDuplicates = actions.Duplicates.Remove;
    result.DisablePush = actions.DisablePush;
    result.StoreAsDeleted = actions.Original.StoreAsDeleted || (actions.RulesApplied ? actions.RulesApplied->StoreAsDeleted : false);
    result.Imap = Imap;
    return result;
}

inline NMdb::ENoSuchFolderAction BuildNoSuchFolderAction(const TActions& actions) {
    std::string val;

    if (actions.RulesApplied && actions.RulesApplied->NoSuchFolder) {
        val = *actions.RulesApplied->NoSuchFolder;
    }

    if (val.empty() && actions.Original.NoSuchFolder) {
        val = *actions.Original.NoSuchFolder;
    }

    if (val.empty()) {
        return NMdb::ENoSuchFolderAction::FallbackToInbox;
    }

    if (val == "fail") {
        return NMdb::ENoSuchFolderAction::Fail;
    } else if (val == "fallback_to_inbox") {
        return NMdb::ENoSuchFolderAction::FallbackToInbox;
    } else if (val == "create") {
        return NMdb::ENoSuchFolderAction::Create;
    }

    throw std::runtime_error("unknown 'on no such folder' action '" + val + "'");
}

inline TMaybeFail<NMdb::TFilterActions> BuildFilterActions(
    const TActions& actions,
    const TRequestFolders& folders,
    const std::optional<TRequestAddedLids>& addedLids,
    const std::optional<TRequestAddedSymbols>& addedSymbols)
{
    TMaybeFail<NMdb::TFilterActions> result;
    if (!actions.UseFilters) {
        return result;
    }
    result = NMdb::TFilterActions();

    std::optional<NMdb::TFolderCoords> originalFolder;
    if (folders.Original) {
        originalFolder = BuildFolder(*folders.Original);
        if (originalFolder) {
            result->OriginalFolder = std::move(*originalFolder);
        }
    }
    if (!originalFolder) {
        result->OriginalFolder = NMdb::TFolderCoords();
        result->OriginalFolder.Path.emplace_back("\\Inbox");
    }
    result->OriginalStoreAsDeleted = actions.Original.StoreAsDeleted;

    if (addedLids) {
        result->AddedLids = *addedLids;
    }

    if (addedSymbols) {
        result->AddedSymbols = *addedSymbols;
    }

    return result;
}

}
