#include "web.h"

#include <mail/mdbsave/lib/web/handlers/base.h>
#include <mail/mdbsave/lib/web/handlers/save.h>

#include <ymod_webserver/server.h>
#include <yplatform/module_registration.h>
#include <yplatform/find.h>

namespace NMdb::NWeb {

void TWebServer::init(const yplatform::ptree& config) {
    auto reactor = yplatform::find_reactor<std::shared_ptr>(config.get<std::string>("reactor"));
    if (!reactor->plain()) {
        throw std::runtime_error("TWebServer requires _pool_count == N and _io_threads == 1");
    }

    auto tvmGuard = std::make_shared<TTvmGuard>(tvm_guard::init(
        config.get_child("tvm_guard"),
        yplatform::find<TTvmModule, std::shared_ptr>("tvm")));

    auto webserver {yplatform::find<TYmodWebServer, std::shared_ptr>("web_server")};

    BindPing(webserver);
    BindSave(tvmGuard, webserver, reactor);
}

void TWebServer::BindPing(TYmodWebServerPtr webserver) const {
    webserver->bind({}, {"/ping"}, [](auto stream) {
        stream->set_code(EHttpCode::ok);
        stream->result_body("pong");
    });
}

void TWebServer::BindSave(TTvmGuardPtr tvmGuard, TYmodWebServerPtr webserver, TReactorPtr reactor) const {
    auto save = NHandler::TBase {
        std::move(tvmGuard),
        "/1/save",
        EHttpMethod::mth_post,
        std::make_shared<NHandler::TSave>()};

    using ymod_webserver::transformer;
    using ymod_webserver::argument;
    webserver->bind(
        {},
        {save.GetUri()},
        [=](auto&& stream, auto&& sessionId) {
            auto wrapper = boost::asio::bind_executor(*reactor->io(), save);
            wrapper(std::move(stream), std::move(sessionId));
        },
        transformer(argument<std::string>("session_id")));
}

}

DEFINE_SERVICE_OBJECT(NMdb::NWeb::TWebServer)
