#include <mdb/context.h>
#include <mdb/mdb_module.h>
#include <mdb/resolve_folder_op.h>

#include <macs/folder_set.h>
#include <macs/folder_factory.h>

#include <gtest/gtest.h>

#include <util/generic/algorithm.h>

namespace NMdb {

struct TFoldersRepositoryFake {
    using TFillFactory = std::function<void(macs::FolderFactory&)>;
    using TFid = std::string;

    template <typename THandler>
    void getAllFoldersWithHidden(THandler&& handler) {
        handler(mail_errors::error_code(), macs::FolderSet(Folders));
    }

    macs::Folder addFolder(const TFillFactory& fill = TFillFactory()) {
        auto fid = std::to_string(NextFid++);

        macs::FolderFactory factory;
        factory.fid(fid);
        factory.name("folder_" + fid);
        factory.parentId(macs::Folder::noParent);
        if (fill) {
            fill(factory);
        }

        Folders.insert({fid, factory.product()});
        return factory.product();
    }

    template <typename THandler>
    void createFolderByPath(const macs::Folder::Path& path, THandler&& handler) {
        auto curFid = macs::Folder::noParent;
        macs::Folder lastFolder;
        for (const auto& name: path) {
            auto foldIt = FindIf(Folders,
                [name, curFid](const auto& item) {
                    return std::tie(curFid, name) == std::tie(item.second.parentId(), item.second.name());
                });
            if (foldIt == Folders.end()) {
                lastFolder = addFolder(
                    [curFid, name](auto& factory) {
                        factory.name(name);
                        factory.parentId(curFid);
                    });
                curFid = lastFolder.fid();
            } else {
                lastFolder = foldIt->second;
                curFid = lastFolder.fid();
            }
        }

        handler(mail_errors::error_code(), lastFolder);
    }

    template <typename THandler>
    void createFolder(const std::string& name, const TFid& parentFid, const macs::Folder::Symbol& symbol, THandler&& handler) {
        auto folderIt = FindIf(Folders,
            [&symbol](const auto& item) {
                return symbol == item.second.symbolicName();
            });
        if (folderIt != Folders.end()) {
            return handler(mail_errors::error_code(), folderIt->second);
        }

        auto lastFolder = addFolder(
            [=](auto& factory) {
                factory.name(name);
                factory.symbol(symbol);
                factory.parentId(parentFid);
            });
        handler(mail_errors::error_code(), lastFolder);
    }

    std::string getLastAddedFid() const {
        return std::to_string(NextFid - 1);
    }

    std::map<TFid, const macs::Folder> Folders;
    int NextFid = 1;
};

struct TResolveFolderOpTest: public ::testing::Test {
    std::pair<boost::system::error_code, std::string> ResolveFolder(const TFolderCoords& coords, ENoSuchFolderAction noSuchFolderAction) {
        boost::system::error_code err;
        std::string fid;
        auto resolveFolderOp = std::make_shared<TResolveFolderOp<TFoldersRepositoryFake*, TResolveDestination>>(
            boost::make_shared<TContext>("", yplatform::log::source()),
            &FoldersRepository,
            coords,
            noSuchFolderAction,
            TResolveDestination{},
            [&err, &fid](auto ec, auto res, auto) {
                err = ec;
                fid = res.Fid;
            });
        yplatform::spawn(resolveFolderOp);
        return {err, fid};
    }

    TFoldersRepositoryFake FoldersRepository;
};

TEST_F(TResolveFolderOpTest, notExistingFolderByFid) {
    TFolderCoords coords;
    coords.Fid = "123";

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_EQ(err, EError::NoSuchFolderPerm);
}

TEST_F(TResolveFolderOpTest, notExistingFolderByFidWithFallback) {
    auto inboxFolder = FoldersRepository.addFolder([&](auto& factory) {
        factory.name("Inbox");
        factory.symbol(macs::Folder::Symbol::inbox);
    });

    TFolderCoords coords;
    coords.Fid = "123";

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::FallbackToInbox);
    EXPECT_FALSE(err);
    EXPECT_EQ(resolvedFid, inboxFolder.fid());
}

TEST_F(TResolveFolderOpTest, notExistingFolderByPath) {
    TFolderCoords coords;
    coords.Path.push_back("not_existing_folder");

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_EQ(err, EError::NoSuchFolderPerm);
}

TEST_F(TResolveFolderOpTest, notExistingFolderByPathCreateFolders) {
    TFolderCoords coords;
    coords.Path.push_back("not_existing_folder");

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Create);
    EXPECT_FALSE(err);
    EXPECT_EQ(resolvedFid, FoldersRepository.getLastAddedFid());
}

TEST_F(TResolveFolderOpTest, notExistingFolderBySymbol) {
    TFolderCoords coords;
    coords.Path.push_back("\\" + macs::Folder::Symbol::inbox.title());

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_EQ(err, EError::NoSuchFolderPerm);
}

TEST_F(TResolveFolderOpTest, emptyFolderPath) {
    TFolderCoords coords;

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_EQ(err, EError::NoSuchFolderPerm);
}

TEST_F(TResolveFolderOpTest, invalidFolderPath) {
    TFolderCoords coords;
    coords.Path.push_back("");

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_EQ(err, EError::NoSuchFolderPerm);
}

TEST_F(TResolveFolderOpTest, shouldResolveFolderByFid) {
    auto folder = FoldersRepository.addFolder();

    TFolderCoords coords;
    coords.Fid = folder.fid();

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_FALSE(err);
    EXPECT_EQ(resolvedFid, folder.fid());
}

TEST_F(TResolveFolderOpTest, shouldResolveFolderByPath) {
    auto folder = FoldersRepository.addFolder();

    TFolderCoords coords;
    coords.Path.push_back(folder.name());

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_FALSE(err);
    EXPECT_EQ(resolvedFid, folder.fid());
}

TEST_F(TResolveFolderOpTest, shouldResolveFolderBySymbol) {
    auto folder = FoldersRepository.addFolder([](auto& factory) {
        factory.name(macs::Folder::Symbol::outbox.title());
        factory.symbol(macs::Folder::Symbol::outbox);
    });

    TFolderCoords coords;
    coords.Path.push_back("\\" + folder.symbolicName().title());

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_FALSE(err);
    EXPECT_EQ(resolvedFid, folder.fid());
}

TEST_F(TResolveFolderOpTest, shouldResolveFolderWithParent) {
    auto parentFolder = FoldersRepository.addFolder();
    auto childFolder = FoldersRepository.addFolder([&](auto& factory) {
        factory.parentId(parentFolder.fid());
    });

    TFolderCoords coords;
    coords.Path.push_back(parentFolder.name());
    coords.Path.push_back(childFolder.name());

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Fail);
    EXPECT_FALSE(err);
    EXPECT_EQ(resolvedFid, childFolder.fid());
}

TEST_F(TResolveFolderOpTest, shouldCreateFolderWithParent) {
    auto parentFolder = FoldersRepository.addFolder();

    TFolderCoords coords;
    coords.Path.push_back(parentFolder.name());
    const std::string subfolder{"subfolder"};
    coords.Path.push_back(subfolder);

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Create);
    EXPECT_FALSE(err);

    auto lastAddedFid = FoldersRepository.getLastAddedFid();
    EXPECT_EQ(resolvedFid, lastAddedFid);
    auto repoSubFolder = FoldersRepository.Folders[lastAddedFid];
    EXPECT_EQ(repoSubFolder.name(), subfolder);
    EXPECT_EQ(repoSubFolder.parentId(), parentFolder.fid());
}

TEST_F(TResolveFolderOpTest, shouldCreateFolderWithChangeableSymbol) {
    TFolderCoords coords;
    coords.Path.push_back("\\Archive");

    auto [err, resolvedFid] = ResolveFolder(coords, ENoSuchFolderAction::Create);
    EXPECT_FALSE(err);

    auto lastAddedFid = FoldersRepository.getLastAddedFid();
    EXPECT_EQ(resolvedFid, lastAddedFid);
    const auto& resolvedSymbol = FoldersRepository.Folders[resolvedFid].symbolicName();
    EXPECT_EQ(resolvedSymbol, macs::Folder::Symbol::archive);
}

} // namespace NMdb
