#include <mdb/context.h>
#include <mdb/mdb_module.h>
#include <mdb/resolve_labels_op.h>

#include <macs/label_factory.h>
#include <macs/label_set.h>

#include <gtest/gtest.h>

namespace NMdb {

struct TLabelsRepositoryFake {
    using TLabelSet = macs::LabelSet;
    using TFillFactory = std::function<void(macs::LabelFactory&)>;

    template <typename THandler>
    void getAllLabels(THandler&& handler) {
        handler(mail_errors::error_code(), Labels);
    }

    template <typename THandler>
    void getOrCreateLabel(const macs::Label::Symbol& symbol, THandler&& handler) {
        for (auto& pair: Labels) {
            auto& label = pair.second;
            if (label.symbolicName() == symbol) {
                return handler(mail_errors::error_code(), label);
            }
        }
        auto label = addLabel([symbol] (auto& factory) {
            factory.symbol(symbol);
        });
        handler(mail_errors::error_code(), label);
    }

    template <typename THandler>
    void getOrCreateLabel(
        const std::string& name,
        const std::string& color,
        const macs::Label::Type& type,
        THandler&& handler)
    {
        for (auto& pair: Labels) {
            auto& label = pair.second;
            if (label.type() == type && label.name() == name) {
                return handler(mail_errors::error_code(), label);
            }
        }
        auto label = addLabel([name, color, type] (auto& factory) {
            factory.name(name);
            factory.color(color);
            factory.type(type);
        });
        handler(mail_errors::error_code(), label);
    }

    macs::Label addLabel(const TFillFactory& fill = TFillFactory()) {
        ++NextLid;

        macs::LabelFactory factory;
        factory.lid(std::to_string(NextLid));
        if (fill) {
            fill(factory);
        }
        auto label = factory.product();

        Labels[label.lid()] = label;
        return label;
    }

    TLabelSet Labels;
    int NextLid = 0;
};

struct TResolveLabelsOpTest: public ::testing::Test {
    std::vector<TResolvedLabel> resolveLabels(
        const std::vector<TLid>& lids,
        const std::vector<TLabelSymbol>& symbols,
        const std::vector<TLabel>& labels,
        const TResolvedFolder& folder = TResolvedFolder())
    {
        std::vector<TResolvedLabel> ret;
        auto resolveLabelsOp = std::make_shared<TResolveLabelsOp<TLabelsRepositoryFake*>>(
            boost::make_shared<TContext>("", yplatform::log::source()),
            &LabelsRepository,
            lids,
            symbols,
            labels,
            folder,
            [&ret] (boost::system::error_code err, const std::vector<TResolvedLabel>& res) {
                ASSERT_FALSE(err);
                ret = res;}
        );
        yplatform::spawn(resolveLabelsOp);
        return ret;
    }

    TLabelsRepositoryFake LabelsRepository;
};

TEST_F(TResolveLabelsOpTest, shouldReturnOnlyExistingLids) {
    auto label = LabelsRepository.addLabel();

    auto resolvedLabels = resolveLabels({label.lid(), "123456789"}, {}, {});
    std::vector<TLid> resolvedLids;
    std::transform(resolvedLabels.begin(), resolvedLabels.end(), std::back_inserter(resolvedLids),
        [](const auto& label) { return label.Lid; });

    std::vector<TLid> expected = {label.lid()};
    EXPECT_EQ(resolvedLids, expected);
}

TEST_F(TResolveLabelsOpTest, shouldResolveSymbolsToLids) {
    auto symbol = macs::Label::Symbol::seen_label;
    auto label = LabelsRepository.addLabel([symbol](auto& factory) {
        factory.symbol(symbol);
    });

    auto resolvedLabels = resolveLabels({}, {symbol.title()}, {});
    std::vector<TLid> resolvedLids;
    std::transform(resolvedLabels.begin(), resolvedLabels.end(), std::back_inserter(resolvedLids),
        [](const auto& label) { return label.Lid; });

    std::vector<TLid> expected = {label.lid()};
    EXPECT_EQ(resolvedLids, expected);
}

TEST_F(TResolveLabelsOpTest, shouldAddSpamLabelForSpamFolder) {
    auto symbol = macs::Label::Symbol::spam_label;
    auto label = LabelsRepository.addLabel([symbol](auto& factory) {
        factory.symbol(symbol);
    });

    auto resolvedLabels = resolveLabels({}, {}, {}, {"", "", "", macs::Folder::Symbol::spam.code()});
    std::vector<TLid> resolvedLids;
    std::transform(resolvedLabels.begin(), resolvedLabels.end(), std::back_inserter(resolvedLids),
        [](const auto& label) { return label.Lid; });

    std::vector<TLid> expected = {label.lid()};
    EXPECT_EQ(resolvedLids, expected);
}

TEST_F(TResolveLabelsOpTest, shouldResolveLabelsToLids) {
    auto name = "test_label";
    auto type = macs::Label::Type::system;
    auto label = LabelsRepository.addLabel([name, type](auto& factory) {
        factory
            .name(name)
            .type(type);
    });

    auto resolvedLabels = resolveLabels({}, {}, {{name, type.title(), ""}});
    std::vector<TLid> resolvedLids;
    std::transform(resolvedLabels.begin(), resolvedLabels.end(), std::back_inserter(resolvedLids),
        [](const auto& label) { return label.Lid; });

    std::vector<TLid> expected = {label.lid()};
    EXPECT_EQ(resolvedLids, expected);
}

TEST_F(TResolveLabelsOpTest, shouldReturnUniqueLids) {
    auto name = "test_label";
    auto type = macs::Label::Type::system;
    auto symbol = macs::Label::Symbol::seen_label;
    auto label = LabelsRepository.addLabel([name, type, symbol](auto& factory) {
        factory
            .name(name)
            .type(type)
            .symbol(symbol);
    });

    auto resolvedLabels = resolveLabels({label.lid(), label.lid()}, {symbol.title()}, {{name, type.title(), ""}});
    std::vector<TLid> resolvedLids;
    std::transform(resolvedLabels.begin(), resolvedLabels.end(), std::back_inserter(resolvedLids),
        [](const auto& label) { return label.Lid; });

    std::vector<TLid> expected = {label.lid()};
    EXPECT_EQ(resolvedLids, expected);
}

// TODO: label symbols test

} // namespace NMdb
