#include <mdb/context.h>

#include <mocks/tabs_repository_mock.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

using namespace testing;
using namespace NMdb;
using namespace macs;

struct TResolveTabsOpTest : public Test {
    TTabsRepositoryMockPtr TabsRepository {std::make_shared<StrictMock<TTabsRepositoryMock>>()};
    const Tab::Type ExistingTabType {Tab::Type::news};
    const Tab::Type RelevantTabType {Tab::Type::relevant};
    const TMaybeFail<TTab> ExistingTabTypeName {"news"};
    const TMaybeFail<TTab> NotExistingTabTypeName {"social"};
    const TResolvedFolder InboxFolder {"1", "inbox", "system", 1};
    const TResolvedFolder DraftsFolder {"5", "drafts", "system", 5};

    using TResolveOp = TResolveTabOp<TTabsRepositoryMockPtr>;
    void ResolveTabs(TResolveOp::THandler handler, const TResolvedFolder& folder, const TMaybeFail<TTab>& tabType = "news") {
        auto resolveTabOp = std::make_shared<TResolveOp>(boost::make_shared<TContext>("", yplatform::log::source()), TabsRepository, folder, tabType, handler);
        yplatform::spawn(resolveTabOp);
    };

    TabSet MakeTabSet() {
        TabsMap tabs;
        Tab tab = TabFactory().type(ExistingTabType).release();
        tabs.insert(std::make_pair(tab.type(), tab));
        return TabSet(tabs);
    }
};

TEST_F(TResolveTabsOpTest, ForGetAllTabsReturnErrorWillReturnError) {
    auto expectedEc = macs::error::logic;
    auto cb = [expectedEc](boost::system::error_code ec, const std::optional<Tab::Type>&) {
        EXPECT_EQ(ec, expectedEc);
    };

    EXPECT_CALL(*TabsRepository, getAllTabs(_))
        .WillOnce(InvokeArgument<0>(mail_errors::error_code(expectedEc), TabSet()));
    ResolveTabs(cb, InboxFolder);
}

TEST_F(TResolveTabsOpTest, ForThrowExceptioWillReturnOperationExceptionError) {
    auto cb = [](boost::system::error_code ec, const std::optional<Tab::Type>&) {
        EXPECT_EQ(ec, EError::OperationException);
    };

    EXPECT_CALL(*TabsRepository, getAllTabs(_))
        .WillOnce(Throw(std::runtime_error("exception")));
    ResolveTabs(cb, InboxFolder);
}

TEST_F(TResolveTabsOpTest, ForNotInboxWillReturnNullOptional) {
    auto cb = [](boost::system::error_code ec, const std::optional<Tab::Type>& tab) {
        EXPECT_FALSE(ec);
        EXPECT_FALSE(tab);
    };

    ResolveTabs(cb, DraftsFolder);
}

TEST_F(TResolveTabsOpTest, ForInboxAndExistTabNameWillReturnTab) {
    auto cb = [this](boost::system::error_code ec, const std::optional<Tab::Type>& tab) {
        EXPECT_FALSE(ec);
        ASSERT_TRUE(tab);
        EXPECT_EQ(*tab, ExistingTabType);
    };

    EXPECT_CALL(*TabsRepository, getAllTabs(_))
        .WillOnce(InvokeArgument<0>(mail_errors::error_code(), MakeTabSet()));
    ResolveTabs(cb, InboxFolder, ExistingTabTypeName);
}

TEST_F(TResolveTabsOpTest, ForInboxAndNotExistTabNameWillReturnTabTypeRelevant) {
    auto cb = [this](boost::system::error_code ec, const std::optional<Tab::Type>& tab) {
        EXPECT_FALSE(ec);
        ASSERT_TRUE(tab);
        EXPECT_EQ(*tab, RelevantTabType);
    };

    EXPECT_CALL(*TabsRepository, getAllTabs(_))
        .WillOnce(InvokeArgument<0>(mail_errors::error_code(), MakeTabSet()));
    ResolveTabs(cb, InboxFolder, NotExistingTabTypeName);
}

TEST_F(TResolveTabsOpTest, ForInboxAndNullTabNameWillReturnTabTypeRelevant) {
    auto cb = [this](boost::system::error_code ec, const std::optional<Tab::Type>& tab) {
        EXPECT_FALSE(ec);
        ASSERT_TRUE(tab);
        EXPECT_EQ(*tab, RelevantTabType);
    };

    ResolveTabs(cb, InboxFolder, {});
}
