package ru.yandex.qe.mail.meetings.api.resource;

import java.io.IOException;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import ru.yandex.qe.mail.meetings.api.resource.dto.AddResourceRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.MergeEventRequest;
import ru.yandex.qe.mail.meetings.api.resource.dto.SwapResourceRequest;

import static ru.yandex.qe.mail.meetings.api.resource.ExportApiService.APPLICATION_JSON_WITH_UTF;

/**
 * API for event smart modification
 *
 * @author Sergey Galyamichev
 */
@Path("/actions")
@Consumes(APPLICATION_JSON_WITH_UTF)
@Produces(CalendarActions.TEXT_HTML_UTF)
public interface CalendarActions {
    String TEXT_HTML_UTF = MediaType.TEXT_HTML + "; " + MediaType.CHARSET_PARAMETER + "=" + "UTF-8";
    /**
     * Get list of scheduled action
     *
     * @return the list
     */
    @GET
    @Path("list/pending")
    List<CalendarAction> pendingActions();

    /**
     * Get waiting list for resources
     *
     * @return the list
     */
    @GET
    @Path("list/search")
    String getAddResourceActions(@QueryParam("login") String login);

    /**
     * Gets action status by action id
     *
     * @return the list
     */
    @GET
    @Path("status")
    String getStatus(@QueryParam("actionId") String actionId);

    /**
     * Tries to find resource for event
     *
     * @param eventId         to be changed
     * @param officeId        office to find resource
     * @param notifyBefore    time interval in minutes to notify organizer that resource can't be found
     * @param isAuto          auto-detect required offices using attendees list
     * @param filter          if present used for as filter for resource to be searched
     * @param login           user login to notify or event organizer if present
     * @param eventDate GMT instant time of event in series
     * @return registered action id
     */
    @GET
    @Path("resource/find")
    String findResource(@QueryParam("eventId") int eventId,
                        @QueryParam("officeId") String officeId,
                        @QueryParam("notifyBefore") int notifyBefore,
                        @QueryParam("isAuto") Boolean isAuto,
                        @Nullable @QueryParam("filter") String filter,
                        @Nullable @QueryParam("login") String login,
                        @Nullable @QueryParam("eventDate") String eventDate);

    /**
     * Tries to find resource for event
     **/
    @POST
    @Path("resource/form/find")
    String findResource(@Nonnull @HeaderParam("login") String organizer,
                        @Nonnull AddResourceRequest request) throws IOException;

    /**
     * Tries to schedule resource swapping for event
     **/
    @POST
    @Path("resource/form/swap")
    String swapResource(@Nonnull @HeaderParam("login") String organizer,
                        @Nonnull SwapResourceRequest request) throws IOException;

    /**
     * Tries to merge to sequential events
     **/
    @POST
    @Path("resource/form/merge")
    void mergeEvent(@Nonnull @HeaderParam("login") String organizer,
                    @Nonnull MergeEventRequest mergeEventRequest);
    /**
     * Cancels schedule scan for resource by id
     *
     * @param actionId action id to cancel
     * @return true if canceled
     */
    @GET
    @Path("cancel")
    String cancelAction(@QueryParam("actionId") String actionId);

    /**
     * Executes action by id
     *
     * @param actionId action id to execute
     * @return human readable result
     */
    @GET
    @Path("execute")
    String executeAction(@QueryParam("actionId") String actionId);


    /**
     * Cancel group of actions by id
     *
     * @param groupId group id to execute
     * @return human readable result
     */
    @GET
    @Path("cancelGroup")
    String cancelGroup(@QueryParam("groupId") String groupId);


    /**
     * Executes group of actions by id
     *
     * @param groupId group id to execute
     * @return human readable result
     */
    @GET
    @Path("executeGroup")
    String executeGroup(@QueryParam("groupId") String groupId);

    @GET
    @Path("ping")
    String ping();
}
