package ru.yandex.qe.mail.analytics.graph;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.IOUtils;

/**
 * @author Sergey Galyamichev
 */
public class Parser {
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static int id = 0;
    private static Map<String, String> clusters = new HashMap<>();

    public static void main(String[] args) throws Exception {
        StringBuilder result = new StringBuilder();
        printHeaders(result);
        InputStream resourceAsStream = Parser.class.getResourceAsStream("/yt___statbox_statbox_dict_last_demands.json");
        List<String> string =  IOUtils.readLines(resourceAsStream, StandardCharsets.UTF_8);
        Jobs jobs = MAPPER.readValue(string.get(0), Jobs.class);
        List<Job> jobList = jobs.getJobs().entrySet().stream()
                .filter(e -> e.getKey().contains("mailan"))
                .map(Map.Entry::getValue)
                .collect(Collectors.toList());
        Set<String> baseNodes = new HashSet<>();
        Set<String> allNodes = new HashSet<>();
        Set<String> reports = new HashSet<>();
        for (Job job : jobList){
            baseNodes.addAll(job.getSrc().keySet());
            allNodes.addAll(job.getDst().keySet());
            reports.addAll(job.getReports());
        }
        baseNodes.removeAll(allNodes);
        allNodes.addAll(baseNodes);
        List<String> sorted = new ArrayList<>(allNodes);

        sorted.sort(String::compareTo);
        for (String node : sorted) {
            result.append(formatTable(node)).append('\n');
        }
        for (String report : reports) {
            result.append(formatReport(report)).append('\n');
        }

        for (Job job : jobList) {
            result.append(formatSubgraph(job)).append('\n');
        }

        for(Job job : jobList) {
            Optional<String> target = job.getDst().keySet().stream()
                    .findFirst();
            if (target.isPresent()) {
                for (String src : job.getSrc().keySet()) {
                    if (!src.equals(target.get())) {
                        result.append(linkToCluster(src, target.get(), job)).append('\n');
                    }
                }
            }
        }
        for(Job job : jobList) {
            Optional<String> source = job.getSrc().keySet().stream()
                    .findFirst();
            if (source.isPresent()) {
                for (String report : job.getReports()) {
                    result.append(linkToReport(source.get(), report, job)).append('\n');
                }
            }
        }
        printTrail(result);
        System.out.println(result);
    }

    private static String linkToCluster(String src, String target, Job job) {
        String label = getLast(job.getJobName());
        return getTableId(src) + "-> " + getTableId(target) + " [lhead=" + getCluster(job) + " label=\"" + label + "\" URL=\"https://st.yandex-team.ru/" + label + "\" fontcolor=blue]";
    }

    private static String linkToReport(String src, String target, Job job) {
        return getTableId(src) + "-> " + getReportId(target) + " [ltail="+ getCluster(job)+ "]";
    }

    private static void printHeaders(StringBuilder result) {
        result.append("%%(graphviz dot)\n" +
                "digraph A { \n" +
                "rankdir=LR; size=\"20,20\";\n" +
                "compound=true;\n");
    }
    private static void printTrail(StringBuilder result) {
        result.append("}\n" +
                "%%\n");
    }
    private static String removeLast(String string) {
        return string.substring(0, string.lastIndexOf("/"));
    }

    private static String getLast(String string) {
        return string.substring(string.lastIndexOf("/") + 1);
    }

    private static String getCluster(Job job) {
        return clusters.computeIfAbsent(job.getJobName(), k -> "cluster" + id++);
    }

    private static String formatSubgraph(Job job) {
        String clusterId = getCluster(job);
        //report_us_metrics [label="Mail/Audience/Features/UserScenarioMetrics"; URL="https://stat.yandex-team.ru/Mail/Audience/Features/UserScenarioMetrics"];
        String name = getLast(job.getJobName());
        clusters.put(name, clusterId) ;
        String nodes = job.getDst().keySet().stream()
                .map(Parser::getTableId)
                .collect(Collectors.joining("; "));
        return String.format("subgraph %s {label=\"%s\"; %s}",  clusterId, name, nodes);
    }

    private static String getTableId(String path) {
        return "table_" + getTableName(path).toLowerCase().replaceAll("-","_");
    }

    private static String getTableName(String path) {
        return getLast(getRemoveDates(path)
                .replaceAll("/1d", ""));
    }

    private static String getRemoveDates(String path) {
        return path.replace("/__ISO_DATE__", "")
                .replaceAll("-__ISO_DATE__", "");
    }

    private static String formatTable(String path) {
        //report_us_metrics [label="Mail/Audience/Features/UserScenarioMetrics"; URL="https://stat.yandex-team.ru/Mail/Audience/Features/UserScenarioMetrics"];
        String id = getTableId(path);
        String name = getTableName(path);
        return String.format("%s [label=\"%s\"; shape = box; URL=\"https://yt.yandex-team.ru/hahn/navigation?path=%s\"];", id, name, getRemoveDates(path));
    }

    private static String getReportId(String path) {
        return "report_" + getLast(removeLast(path)).toLowerCase().replaceAll("-","_");
    }

    private static String getReportName(String path) {
        return removeLast(path);
    }

    private static String formatReport(String path) {
        //report_us_metrics [label="Mail/Audience/Features/UserScenarioMetrics"; URL="https://stat.yandex-team.ru/Mail/Audience/Features/UserScenarioMetrics"];
        String name = getReportName(path);
        String reportId = getReportId(path);
        return String.format("%s [label=\"%s\"; URL=\"https://stat.yandex-team.ru/%s\"];", reportId,  name, path);
    }
}

