package ru.yandex.qe.mail.meetings.analisys;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventDate;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventUser;
import ru.yandex.qe.mail.meetings.services.calendar.dto.User;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;
import ru.yandex.qe.mail.meetings.ws.EventObject;

public class DismissedUserAnalyzer {
    private static final Logger LOG = LoggerFactory.getLogger(DismissedUserAnalyzer.class);

    private final CalendarHelper calendar;
    private final UserCheck userCheck;

    public DismissedUserAnalyzer(CalendarWeb calendarWeb, UserCheck userCheck) {
        this.calendar = new CalendarHelper(calendarWeb);
        this.userCheck = userCheck;
    }

    public Pair<Event, List<User>> findAttendees(EventObject eventObject) {
        try {
            Integer eventId = eventObject.getEvent().getEventId();
            Event event = calendar.getBaseEvent(eventId);
            Set<User> eventUsers = new HashSet<>(event.getAttendees());
            eventUsers.addAll(event.getSubscribers());
            List<User> dismissed = eventUsers.stream()
                    .filter(this::checkUser)
                    .collect(Collectors.toList());
            return dismissed.isEmpty() ? null : Pair.of(event, dismissed);
        } catch (Exception e) {
            Optional<Integer> id = Optional.ofNullable(eventObject)
                    .map(EventObject::getEvent)
                    .map(EventDate::getEventId);
            LOG.warn("Unable to get event with id = [{}] info due to the exception", id.orElse(null), e);
        }
        return null;
    }

    public Pair<Event, InspectionsResult> check(EventObject eventObject) {
        try {
            InspectionsResult result = new InspectionsResult();
            Integer eventId = eventObject.getEvent().getEventId();
            Event event = calendar.getBaseEvent(eventId);
            EventUser organizer = event.getOrganizer();
            if (organizer != null && checkUser(organizer)) {
                result.addCritical(InspectedItem.critical(eventObject.getEvent(), event, organizer,
                        String.format("Event has dismissed organizer and is editable = %b", event.isParticipantsCanEdit())));
            }
            List<EventUser> real = event.getAttendees().stream()
                    .filter(this::checkUser)
                    .collect(Collectors.toList());
            if (real.size() > 0 && event.getAttendees().size() == 1) {
                result.addCritical(InspectedItem.critical(eventObject.getEvent(), event,
                        String.format("Event has dismissed users: %s of %d members", getLogins(real), event.getAttendees().size())));
            } else if (real.size() > 0) {
                result.addWarning(InspectedItem.warn(eventObject.getEvent(), event,
                        String.format("Event has dismissed users: %s of %d members", getLogins(real), event.getAttendees().size())));
            }
            return result.getAll().isEmpty() ? null : Pair.of(event, result);
        } catch (Exception e) {
            LOG.warn("Unable to get event info due to the exception", e);
        }
        return null;
    }

    private boolean checkUser(User user) {
        return userCheck.isDismissed(user.getLogin());
    }

    private String getLogins(List<EventUser> real) {
        return real.stream()
                .map(EventUser::getLogin)
                .collect(Collectors.joining(", "));
    }
}
