package ru.yandex.qe.mail.meetings.analisys;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Decision;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventDate;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventUser;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gap;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;
import ru.yandex.qe.mail.meetings.ws.EventObject;

/**
 * Tries to take into account info about employment statuses, absences and trips
 *
 * @author Sergey Galyamichev
 */
public class EventGapAnalyzer<R> {
    private static final Logger LOG = LoggerFactory.getLogger(EventGapAnalyzer.class);
    private static final Set<String> IGNORE_USERS = Collections.singleton("info@calendar.yandex-team.ru");

    private static final String IN_TRIP = "IN_TRIP";
    private static final String NO_RESOURCE = "NO_RESOURCE";
    private static final Set<String> SYSTEM_RESOURCES = Set.of(IN_TRIP, NO_RESOURCE);

    private static final int RED_ROSE_ID = 1;
    private static final int MAMONTOV_ID = 147;
    private static final int STROGANOV_ID = 148;
    private static final String TRIP = "trip";

    private final CalendarHelper calendar;
    private final CalendarWeb calendarWeb;
    private final UserCheck userCheck;
    private final Map<String, List<Gap>> gaps;
    private final BiFunction<Event, Event, EventGapsResultBuilder<R>> inspectionFactory;

    public EventGapAnalyzer(CalendarHelper calendar,
                            CalendarWeb calendarWeb,
                            UserCheck userCheck,
                            Map<String, List<Gap>> gaps,
                            BiFunction<Event, Event, EventGapsResultBuilder<R>> inspectionFactory) {
        this.calendar = calendar;
        this.calendarWeb = calendarWeb;
        this.userCheck = userCheck;
        this.gaps = gaps;
        this.inspectionFactory = inspectionFactory;
    }

    /**
     * Since mamontov, stroganov and red rose offices placed very close I consider them as one office
     */
    private static int getResourceId(int resourceId) {
        return resourceId == MAMONTOV_ID || resourceId == STROGANOV_ID ? RED_ROSE_ID : resourceId;
    }

    public static boolean isAGap(Gap gap, Event e) {
        return e.getExternalId() != null && e.getExternalId().startsWith("" + gap.getId());
    }

    public R check(@Nonnull EventObject eventObject) {
        try {
            Integer eventId = eventObject.getEvent().getEventId();
            Resource.Info resourceInfo = calendarWeb.getResourceInfo(eventObject.getResource().getResourceInfo().getEmail());
            if (!resourceInfo.getResourceType().isRoom()) {
                return null;
            }
            Event event = calendarWeb.getEvent(eventId);
            EventGapsResultBuilder<R> inspections = inspectionFactory.apply(event, calendar.getBaseEvent(event));

            Map<Integer, Resource.Info> officeToResource = event.getResources().stream()
                    .collect(Collectors.toMap(r -> getResourceId(r.getOfficeId()), Function.identity(), (a, b) -> {
                        inspections.twoResourcesInOffice(a, b);
                        return a;
                    }));

            Map<String, List<EventUser>> real = event.getResources().stream()
                    .collect(Collectors.toMap(Resource.Info::getEmail, t -> new ArrayList<>()));
            EventUser organizer = event.getOrganizer();
            if (organizer == null) {
                inspections.organizerIsNull();
            } else if (!checkUser(event, officeToResource, real, organizer)) {
                inspections.organizersAbsence(organizer);
            }
            for (EventUser user: event.getAttendees()) {
                checkUser(event, officeToResource, real, user);
            }
            return buildResolution(inspections, eventObject, real);
        } catch (Exception e) {
            LOG.warn("Unable to get event info due to the exception", e);
        }
        return null;
    }

    /**
     * Adds user to one of the following rooms:
     *      - if not in trip and found resource in office then resourceId of the found
     *      - if in trip special room IN_TRIP (means probably will use a resource)
     *      - if user in office without resource and not in a trip to special room NO_RESOURCE (user doesn't requires
     *          resource but probably resource is needed to communicate with him)
     * @param resources - officeId to resource info map
     * @param real - current resource to user split state
     * @param user - user to update real map
     * @return true if real map updated
     */
    private boolean checkUser(Event event, Map<Integer, Resource.Info> resources, Map<String, List<EventUser>> real, EventUser user) {
        if (IGNORE_USERS.contains(user.getEmail())) {
            return false;
        }
        List<Gap> userGaps = gaps.getOrDefault(user.getLogin(), Collections.emptyList());
        int intResourceId = getResourceId(user.getOfficeId() != null ? user.getOfficeId() : 0);
        String resourceId = isInTrip(event, userGaps) ? IN_TRIP :
                resources.containsKey(intResourceId) ?
                resources.get(intResourceId).getEmail() : NO_RESOURCE;

        List<EventUser> visitors = real.computeIfAbsent(resourceId, k -> new ArrayList<>());

        if (!userCheck.isDismissed(user.getLogin()) && !isAbsence(event, userGaps) && willVisit(user)) {
             return visitors.add(user);
        }
        return false;
    }

    private boolean willVisit(EventUser user) {
        return user.getDecision() != Decision.NO;
    }

    private boolean isInTrip(Event event, List<Gap> gaps) {
        return gaps.stream()
                .filter(g -> isIn(g, event))
                .map(Gap::getWorkflow)
                .anyMatch(TRIP::equals);
    }

    public static boolean isIn(Gap gap, EventDate event) {
        return gap.getDateFrom().getTime() <= event.getStart().getTime()
                &&  event.getEnd().getTime() <= gap.getDateTo().getTime();
    }

    private boolean isAbsence(Event event, List<Gap> gaps) {
        return gaps.stream()
                .filter(g -> isIn(g, event))
                .anyMatch(EventGapAnalyzer::isAbsence);
    }

    public static boolean isAbsence(Gap gap) {
        // todo manage resources even if user gonna work during the gap
        return !gap.isWorkInAbsence() && !TRIP.equals(gap.getWorkflow());
    }

    private R buildResolution(EventGapsResultBuilder<R> inspections, EventObject event, Map<String, List<EventUser>> utilization) {
        Integer eventId = event.getEvent().getEventId();
        for (String system : SYSTEM_RESOURCES) {
            if (utilization.containsKey(system) && utilization.get(system).isEmpty()) {
                utilization.remove(system); // who cares...
            }
        }
        int realUsers = utilization.values().stream()
                .mapToInt(Collection::size)
                .sum();
        Set<String> emptyResources = utilization.entrySet().stream()
                .filter(e -> e.getValue().isEmpty())
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
        String resources = String.join(", ", emptyResources);
        if (realUsers == 0) {
            if (resources.isBlank()) {
               LOG.warn("No resources and users for event {}, comes from office {} resource {}",
                       eventId, event.getOffice(), event.getResource().getResourceInfo().getEmail() );
            }
            emptyResources.forEach(inspections::noActiveUsersFound);
        } else {
            if (realUsers == 1) {
                if (utilization.size() == 1) {
                    utilization.keySet().forEach(inspections::oneUserInResource);
                } else {
                    utilization.keySet().forEach(inspections::multiResource);
                }
            } else {
                utilization.forEach((resourceId, users) -> {
                    if (users.isEmpty()) {
                        if (utilization.containsKey(IN_TRIP) && utilization.get(IN_TRIP).size() < emptyResources.size()) {
                            emptyResources.forEach(inspections::probablyEmptyResource);
                        } else {
                            inspections.resourceUtilizationUnknown(resourceId);
                        }
                    }
                });
            }
        }
        return inspections.build();
    }
}
