package ru.yandex.qe.mail.meetings.analisys;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

/**
 * Staff service interaction strategy
 *
 * @author Sergey Galyamichev
 */
public interface UserCheck {
    /**
     * Checks if user employment status
     * @param login to check
     * @return true if user dismissed
     */
    boolean isDismissed(String login);

    /**
     * Loads all dismissed user at once
     */
    class Bulk implements UserCheck {
        private final StaffClient staffClient;
        private Set<String> loginSet;

        public Bulk(StaffClient staffClient) {
            this.staffClient = staffClient;
        }

        @Override
        public boolean isDismissed(String login) {
            if (loginSet == null) {
                loginSet = staffClient.getAll(true).stream()
                        .map(Person::getLogin)
                        .collect(Collectors.toSet());
            }
            return loginSet.contains(login);
        }
    }

    /**
     * Loads users be request
     */
    class OneByOne implements UserCheck {
        private final Map<String, Person> persons = new ConcurrentHashMap<>();
        private final StaffClient staffClient;

        public OneByOne(StaffClient staffClient) {
            this.staffClient = staffClient;
        }

        @Override
        public boolean isDismissed(String login) {
            if (login == null) {
                return true;
            }
            return persons.computeIfAbsent(login, staffClient::getByLogin)
                    .getOfficial().isDismissed();
        }
    }
}
