package ru.yandex.qe.mail.meetings.blamer;

import java.util.Date;
import java.util.List;

import javax.inject.Named;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.stereotype.Repository;

import ru.yandex.qe.mail.meetings.services.calendar.DateParameterConverterProvider;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * The class to access data related to declined messages notification
 *
 * @author Sergey Galiamichev
 */

@Repository
public class DeclinedEventsDao {
    private static final Logger LOG = LoggerFactory.getLogger(DeclinedEventsDao.class);

    // to support nulls in dictionaries
    private static final String DISABLE_STRICT = "PRAGMA yson.DisableStrict;";
    // robot has no access to tmp folder
    private static final String TMP_FOLDER = "PRAGMA yt.TmpFolder = '//home/calendar/corp/raw/tmp'; ";

    private static final String CALENDAR_EVENT_LOG = "//logs/calendar-yt-events-log-json/1d";
    private static final String DISABLE_CACHE =
            "PRAGMA yt.QueryCacheMode='disable'; " +
                    "PRAGMA yt.ReleaseTempData='immediate'; " +
                    "PRAGMA yt.AutoMerge='disabled'; ";
    private final String declinedEventLog;

    private NamedParameterJdbcTemplate jdbcTemplate;

    @Autowired
    public DeclinedEventsDao(@Named("ytDataSource") DataSource ytDataSource, @Value("${calendar.yt.root.path}") String ytPath) {
        this.jdbcTemplate = new NamedParameterJdbcTemplate(ytDataSource);
        this.declinedEventLog = ytPath + "/declines";
    }

    public String getDeclinedEventLog(Date date) {
        return declinedEventLog + "/" + DateParameterConverterProvider.CONVERTER.toString(date);
    }

    /**
     * Gets all declined by display events for a specified date from calendar events common log
     *
     * @param date - date requested
     *
     * @return list of declined event
     */
    public List<DeclineEvent> getDeclineEvent(Date date) {
        String forDate = DateParameterConverterProvider.CONVERTER.toString(date);
        String sql = DISABLE_STRICT + TMP_FOLDER + DISABLE_CACHE +
                "SELECT event_id, resource_email, count(participant) AS attendees FROM ( " +
                    "SELECT " +
                        "Yson::LookupInt64(event_info, 'event_id') AS event_id, " +
                        "Yson::LookupString(event_info, 'from') AS resource_email, " +
                        "Yson::LookupString(event_info, 'to') AS participant " +
                    "FROM " +
                        "RANGE(:table, :date, :date) "+
                    "WHERE " +
                        "action = 'sendMailTask' " +
                        "AND Yson::LookupString(event_info, 'mail_type') == 'RESOURCE_UNCHECKIN'" +
                    ")" +
                "GROUP BY event_id, resource_email;";
        SqlParameterSource source = new MapSqlParameterSource("table", CALENDAR_EVENT_LOG)
                .addValue("date", forDate);
        return jdbcTemplate.query(sql, source, new BeanPropertyRowMapper<>(DeclineEvent.class));
    }

    /**
     * Returns aggregated info for declined by display meetings for a period
     *
     * @param range - date period
     * @param threshold - minimum unconfirmed meeting amount
     *
     * @return list of aggregated
     */
    public List<DeclineEvents> getDeclineEvent(DateRange range, int threshold) {
        String fromDate = DateParameterConverterProvider.CONVERTER.toString(range.getFrom());
        String toDate = DateParameterConverterProvider.CONVERTER.toString(range.getTo());
        LOG.debug("In getDeclineEvent from = {} to {}, threshold = {}", fromDate, toDate, threshold);
        String sql = TMP_FOLDER + DISABLE_CACHE +
                "SELECT " +
                    "email, login, name, COUNT(eventId) as count " +
                "FROM " +
                    "RANGE(:table, :dateFrom, :dateTo) " +
                "GROUP BY " +
                    "email, login, name " +
                "HAVING " +
                    "COUNT(eventId) > :threshold " +
                "ORDER BY count DESC";
        SqlParameterSource source = new MapSqlParameterSource("table", declinedEventLog)
                .addValue("dateFrom", fromDate)
                .addValue("dateTo", toDate)
                .addValue("threshold", threshold);
        return jdbcTemplate.query(sql, source, new BeanPropertyRowMapper<>(DeclineEvents.class));
    }

    /**
     * Get stored declined events
     *
     * @param range - date period
     * @param email - user email
     *
     * @return list of stored declined event
     */
    public List<DeclineEvent> getDeclineEvent(DateRange range, String email) {
        String fromDate = DateParameterConverterProvider.CONVERTER.toString(range.getFrom());
        String toDate = DateParameterConverterProvider.CONVERTER.toString(range.getTo());
        LOG.debug("In getDeclineEvent from = {} to {}, email = {}", fromDate, toDate, email);
        String sql = TMP_FOLDER + DISABLE_CACHE +
                "SELECT " +
                    "eventDate, eventId, resourceEmail, name, WeakField(r.name, 'String') AS resourceName " +
                "FROM (" +
                    "SELECT " +
                        "TableName() AS eventDate, eventId, resourceEmail, name " +
                    "FROM " +
                        "RANGE(:table, :dateFrom, :dateTo) " +
                    "WHERE " +
                        "email = :email) AS t " +
                "LEFT JOIN " +
                    "`//home/calendar/corp/raw/resources/" + fromDate +"` AS r " +
                "ON " +
                    "WeakField(r.email, 'String') = t.resourceEmail " +
                "ORDER BY eventDate";
        SqlParameterSource source = new MapSqlParameterSource("table", declinedEventLog)
                .addValue("dateFrom", fromDate)
                .addValue("dateTo", toDate)
                .addValue("email", email);
        return jdbcTemplate.query(sql, source, new BeanPropertyRowMapper<>(DeclineEvent.class));
    }
}
