package ru.yandex.qe.mail.meetings.booking;

import java.util.HashMap;
import java.util.Map;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import com.fasterxml.jackson.databind.ObjectMapper;
import freemarker.template.Configuration;
import org.apache.commons.lang3.CharEncoding;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

import ru.yandex.qe.mail.meetings.booking.dto.FullSearchResult;
import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;


@Component
public class BookingResultMessageBuilder extends AbstractMessageBuilder {
    private static final Logger LOG = LoggerFactory.getLogger(BookingResultMessageBuilder.class);

    private static final String TEMPLATE = "booking_failed_template.ftl";

    private static final ObjectMapper MAPPER = new ObjectMapper();

    public BookingResultMessageBuilder(Configuration configuration, StaffClient staffClient, NotificationConfiguration notConfig) {
        super(configuration, staffClient, notConfig);
    }


    public void prepareMessage(MimeMessage mimeMessage, String user, Map<String, String> params, FullSearchResult.MissingReason reason, boolean isDebug) throws MessagingException {
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, CharEncoding.UTF_8);
        prepareMessageHeadersDirect(message, user + AT_YANDEX, null);
        message.setSubject(user + "@, " + " бронирование не удалось");
        message.setText(buildBody(user, params, reason, isDebug), true);
    }

    private String buildBody(String user, Map<String, String> params, FullSearchResult.MissingReason reason, boolean isDebug) throws MessagingException {
        try {
            Map<String, Object> valuesMap = new HashMap<>();

            var readableReason = makeReadable(reason);

            var duration = params.getOrDefault("own_duration", "" + params.get("duration"));
            var participants =  params.getOrDefault("participants", "без обязательных участников");
            var additionalParticipants =  params.getOrDefault("additional_attendees", "без дополнительных участников");
            var datesInterval = params.getOrDefault("dates_interval", "" + params.get("dates_exact"));
            var timeInterval = params.get("early_border") + " - " + params.get("late_border");

            valuesMap.put("name", user + "@");
            valuesMap.put("meetingName", "" + params.get("meeting_name"));
            valuesMap.put("description", "" + params.get("meeting_desc"));
            valuesMap.put("duration", "" + duration + " мин.");
            valuesMap.put("participants", participants);
            valuesMap.put("additionalParticipants", additionalParticipants);
            valuesMap.put("readableReason", readableReason);
            valuesMap.put("timeInterval", timeInterval);
            valuesMap.put("datesInterval", datesInterval);

            if (isDebug) {
                valuesMap.put("fullConfig", MAPPER.writeValueAsString(params));
            } else {
                valuesMap.put("fullConfig", "");
            }

            if (reason == FullSearchResult.MissingReason.INTERNAL_ERROR && !isDebug) {
                valuesMap.put("hint", "");
            } else {
                valuesMap.put("hint", "Попробуйте поменять параметры встречи или число участников.");
            }

            LOG.info("preparing template {} with params {}", TEMPLATE, valuesMap);
            return FreeMarkerTemplateUtils.processTemplateIntoString(configuration.getTemplate(TEMPLATE), valuesMap);
        } catch (Exception e) {
            throw new MessagingException("Unable to build body", e);
        }
    }

    private String makeReadable(FullSearchResult.MissingReason reason) {
        if (reason == null) {
            return "причина не установлена, обязательно сообщите нам об этой проблеме";
        }
        switch (reason) {
            case PERSON_HAVE_NO_INTERSECTION:
                return "У обязательных участников нет пересекающихся свободных слотов под встречи";
            case TIME_RESTRICTIONS_ERROR:
                return "Заданный временной интервал слишком узок";
            case NO_FREE_ROOMS:
                return "Нет свободных переговорок, попробуйте разрешить бронирование с отложенным добавлением переговорки";
            case REGULAR_BOOKING_UNAVAILABLE:
                return "Невозможно создать регулярную встречу по вашим параметрам";
            case API_INTERRACTION_ERROR:
                return "Ошибка взаимодействие с API календаря. Попробуйте позже";
            case INTERNAL_ERROR:
                return "Внутренняя ошибка сервиса, мы уже работаем над данной проблемой";
            case NO_OFFICE:
                return "Ни один из выбранных офисов не поддерживается";
            default:
                return "К этой ошибке не написали описание: " + reason;
        }
    }
}
