package ru.yandex.qe.mail.meetings.booking;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.Duration;
import org.joda.time.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.api.resource.CalendarActions;
import ru.yandex.qe.mail.meetings.booking.dto.BookingResult;
import ru.yandex.qe.mail.meetings.booking.dto.FullSearchResult;
import ru.yandex.qe.mail.meetings.booking.dto.OfficeRequirements;
import ru.yandex.qe.mail.meetings.booking.dto.SearchResult;
import ru.yandex.qe.mail.meetings.booking.impl.TimeTableImpl;
import ru.yandex.qe.mail.meetings.booking.util.BookingUtils;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventType;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Intervalable;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Repetition;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Response;
import ru.yandex.qe.mail.meetings.services.calendar.dto.WebEventCreateData;
import ru.yandex.qe.mail.meetings.services.calendar.dto.suggest.SuggestBody;
import ru.yandex.qe.mail.meetings.services.calendar.dto.suggest.SuggestResponse;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.RepetitionUtils;


@Service("fuzzyBookingService")
public class FuzzyBookingService {
    private static final Logger LOG = LoggerFactory.getLogger(FuzzyBookingService.class);

    private static final String AT_YANDEX = "@yandex-team.ru";

    @Nonnull
    private final RoomService roomService;

    @Nonnull
    private final StaffClient staffClient;

    @Nonnull
    private final PersonService personService;

    @Nonnull
    private final CalendarWeb calendarWeb;

    @Nonnull
    private final CalendarUpdate calendarUpdate;

    @Nonnull
    private final CalendarActions calendarActions;


    @Inject
    public FuzzyBookingService(@Nonnull RoomService roomService,
                               @Nonnull StaffClient staffClient,
                               @Nonnull PersonService personService,
                               @Nonnull CalendarWeb calendarWeb,
                               @Nonnull CalendarUpdate calendarUpdate,
                               @Nonnull CalendarActions calendarActions) {
        this.roomService = roomService;
        this.staffClient = staffClient;
        this.personService = personService;
        this.calendarWeb = calendarWeb;
        this.calendarUpdate = calendarUpdate;
        this.calendarActions = calendarActions;
    }


    /**
     * Метод осуществляет реальное бронирование. Поиск слотов и переговорок происходит через `findTimeSlotAndRoomForRegular` & `findTimeSlotAndRoom`
     * @param timeRestrictions - ограничение на время встречи
     * @param duration - продолжительность встречи
     * @param meetingInfo - базовая информация о встрече
     * @param attendees - учатники
     * @param additionalAttendees - не обязательные участники встречи, их занятость не учитывается
     * @param roomRestrictions - пожелания к переговорке
     * @param allowMeetingsWithoutRooms - разрешить отложенное бронирование переговорок
     * @param repetition - параметры повтора для регулярных встреч
     * @return true - успех, false - провал
     */
    public BookingResult bookMeeting(@Nonnull TimeTable timeRestrictions,
                                     @Nonnull Duration duration,
                                     @Nonnull BasicMeetingInfo meetingInfo,
                                     @Nonnull Set<String> attendees,
                                     @Nonnull Set<String> additionalAttendees,
                                     @Nonnull Resource.Info roomRestrictions,
                                     boolean allowMeetingsWithoutRooms,
                                     @Nullable Repetition repetition,
                                     int notifyBefore) {

        var organizer = meetingInfo.organizer();
        LOG.info("booking meeting with org {} for [{} ({})] regular:{}", organizer, String.join(",", attendees), String.join(",", additionalAttendees), Optional.ofNullable(repetition).map(Repetition::toString).orElse("false"));

        var attendeesWithOrganizer = Stream.concat(Stream.of(organizer), attendees.stream()).collect(Collectors.toUnmodifiableSet());
        var fullAttendees = Stream.concat(attendeesWithOrganizer.stream(), additionalAttendees.stream()).collect(Collectors.toUnmodifiableSet());

        final FullSearchResult result = repetition != null ?
                findTimeSlotAndRoomForRegular(
                        timeRestrictions,
                        duration,
                        organizer,
                        attendeesWithOrganizer,
                        additionalAttendees,
                        roomRestrictions,
                        repetition
                ) : findTimeSlotAndRoom(
                        timeRestrictions,
                        duration,
                        organizer,
                        attendeesWithOrganizer,
                        additionalAttendees,
                        roomRestrictions,
                        allowMeetingsWithoutRooms
                );
        LOG.info("RESULT: {} for [timeRestrictions = {}, duration = {}, meetingInfo = {}, attendees = {}, " +
                        "additionalAttendees = {}, roomRestrictions = {}, allowMeetingsWithoutRooms = {}, repetition = {}, " +
                        "notifyBefore = {}]", result, timeRestrictions, duration, meetingInfo, attendees, additionalAttendees,
                roomRestrictions, allowMeetingsWithoutRooms, repetition, notifyBefore);
        //ничего не нашли
        if (result.isEmpty()) {
            LOG.info("results are empty, {}", result.missingReason());
            return BookingResult.empty(result.missingReason());
        }

        boolean wasRegularBooked = false;
        if (result.isRegular()) {
            var regular = result.regular();
            var r = bookSingle(regular, meetingInfo, fullAttendees, repetition);
            wasRegularBooked = r.isOk();
            if (!wasRegularBooked) {
                LOG.info("can't book regular meeting, fail");
                return BookingResult.empty(FullSearchResult.MissingReason.API_INTERRACTION_ERROR);
            }
        }

        var additionalBooked = result.additional().stream().map(sr -> {
            var r = bookSingle(sr, meetingInfo, fullAttendees, null);
            if (!r.isOk()) {
                return false;
            }

            // нехватающие переговорки поставим в очередь
            if (!sr.missingOffices().isEmpty()) {
                String officeIds = idsToString(sr.missingOffices());
                LOG.info("start searching resources for offices {}", officeIds);
                calendarActions.findResource(r.getShowEventId(), officeIds, notifyBefore, false, null, organizer, null);
            }
            return r.isOk();
        })
        // хак что бы вычислить все значения
        .collect(Collectors.toUnmodifiableList()).stream().anyMatch(b->b);

        return (wasRegularBooked || additionalBooked) ? BookingResult.ok() : BookingResult.empty(FullSearchResult.MissingReason.API_INTERRACTION_ERROR);
    }

    public static String idsToString(Set<Integer> officeIds) {
        return officeIds.stream()
                .map(Objects::toString)
                .collect(Collectors.joining(","));
    }

    /**
     * Находим подходящие временные слоты и переговорки для разовой встречи
     * @param timeRestrictions - ограничения по времени: даты поиска и интервалы, в которые встречи ставить нельзя
     * @param duration - продолжитеьлность встречи
     * @param attendees - участиники (логины)
     * @param roomRestrictions - пожелания к переговоркам (вай-фай, проектор и тд). Ограничения на кол-во людей добавляются автоматически
     * @param allowMeetingsWithoutRooms - разрешить бронировать встречу без всех необходимых переговорок
     */
    @Nonnull
    public FullSearchResult findTimeSlotAndRoom(
            @Nonnull TimeTable timeRestrictions,
            @Nonnull Duration duration,
            @Nonnull String organizer,
            @Nonnull Set<String> attendees,
            @Nonnull Set<String> additionalAttendees,
            @Nonnull Resource.Info roomRestrictions,
            boolean allowMeetingsWithoutRooms) {

        var profiles = staffClient.getByLogins(attendees, false);

        var allProfiles = staffClient.getByLogins(
                Stream.concat(
                        attendees.stream(),
                        additionalAttendees.stream()
                ).collect(Collectors.toSet()),
                false);

        // тут для каждого офиса создаем требования по кол-ву людей
        // [PSSP-192] additional attendees must be counted
        var officeReq = createOfficeRequirements(allProfiles);
        if (officeReq.isEmpty()) {
            return FullSearchResult.empty(FullSearchResult.MissingReason.NO_OFFICE);
        }

        // самые широкие из разумных врепенных ограничений
        var terms = timeRestrictions.terms();

        // получаем отображение человек -> расписание
        var personTimetables = personService.timetableForPersonsParallel(profiles, terms.getStart().toDate(), terms.getEnd().toDate());

        // объяединяем все расписания людей и получаем расписание, где свободные слоты обозначают, что все участники могут принять встречу в это время
        var allPersonTimeTable = TimeTableImpl.intersectFree(terms, personTimetables.values());

        // эту проверку можно провести позже, но тут она точно поможет выявить причину неудачи
        if (allPersonTimeTable.freeIntervals(duration).isEmpty()) {
            LOG.info("persons timetable intersection doe not contains interval longer then {}m.", duration.getStandardMinutes());
            return FullSearchResult.empty(FullSearchResult.MissingReason.PERSON_HAVE_NO_INTERSECTION);
        }

        var allPersonTimeTableWR = TimeTableImpl.intersectFree(terms, allPersonTimeTable, timeRestrictions);
        // если нет общих свободных слотов, то ничего предложить не можем
        if (allPersonTimeTableWR.freeIntervals(duration).isEmpty()) {
            LOG.info("empty intersection with time restrictions");
            return FullSearchResult.empty(FullSearchResult.MissingReason.TIME_RESTRICTIONS_ERROR);
        }

        var officeGolden = extendOfficeRequirements(officeReq, roomRestrictions);

        var requiredIds = officeGolden.keySet();

        // получаем отображение офис -> список переговорок, подходящих по параметрам + согласованное расписание (с расписанием всех участников и временнЫми ограничениями) для каждой переговорки
        // затем пересекам с расписанием, которое подходит всем пользователям
        var roomsByOffice = officeReq
                .stream()
                .flatMap(or -> roomService
                        // создаем ограничение вида 'ограничение на оборудование' + 'вместимость' + 'офис' и получаем подходящие переговорки
                        .roomsWithRestrictionsAndOrder(
                                BookingUtils.withOfficeRestriction(
                                        BookingUtils.withCapacityRestriction(roomRestrictions, or.count),
                                        or.officeId
                                )
                        )
                        .stream()
                        // для каждой подходящей переговорки получаем расписание в нужном интервале времени
                        .map(r -> roomService.schedule(r, timeRestrictions.terms()))
                        // оставляем в расписании переговорки только свободные слоты, которые не пересекаются с занятыми слотами участников и подходят под все временные условия
                        .map(rwt -> {
                            // пересечение всех расписаний: всех пользователей, текущей переговорки и временных ограничений
                            var finalTimeTable = TimeTableImpl.intersectFree(terms, rwt.timeTable, timeRestrictions, allPersonTimeTable);
                            return new RoomService.ResourceWithTimeTable(rwt.info, finalTimeTable);
                        }))
                .filter(rwt -> !rwt.timeTable.freeIntervals(duration).isEmpty())
                .collect(Collectors.groupingBy(r -> r.info.getOfficeId(), Collectors.toList()));

        // добьем пустыми списками офисы, где не удалось найти переговорки
        // далее находимся в ситуации, когда не удалось нормально забронировать встречу
        requiredIds.forEach(id -> roomsByOffice.putIfAbsent(id, Collections.emptyList()));

        var result =  BookingUtils.combineAllAndOrder(officeGolden, roomsByOffice, terms)
                .stream()
                .flatMap(p ->
                        // берем все интервалы длитеьностью > duration и превращаем в результат
                        p.getValue()
                                .freeIntervals(duration)
                                .stream()
                                .map(i -> new SearchResult(requiredIds, p.getKey(), i.withDurationAfterStart(duration))
                                )
                )
                .collect(Collectors.toUnmodifiableList());

        LOG.info("got {} meeting candidates", result.size());
        logSearchResults(result,3);

        /**
         * некоторые переговорки могут быть не доступны для бронирования конкретному человеку, их надо выкинуть
         * метод достаточно долгий, т.к. получает расписание переговорок о офисам не из кеша
         * можно было сразу получать расписание переговорок для конкретного человека, а не брать роботное представление из кеша
         * мне кажется, что такое решение будет работать значительно дольше для кейсов, когда у нас нет возможности забпронировать переговорку, но это обсуждаемо
         **/
        result = applyPersonalFilter(result, organizer);

        LOG.info("got {} meeting candidates after personal filter", result.size());
        logSearchResults(result,3);

        // если есть варианты бронирования, то возвращаем их
        if (!result.isEmpty()) {
            LOG.info("full success");
            return FullSearchResult.of(result.get(0));
        }

        // если нельзя создавать встречи без переговорок, то ничего не поделать
        if (!allowMeetingsWithoutRooms) {
            LOG.info("booking without room is not allowed by user, fail");
            return FullSearchResult.empty(FullSearchResult.MissingReason.NO_FREE_ROOMS);
        }

        final Comparator<OfficeRequirements> reqCountCmp =
                Comparator.comparing(or -> -or.count);

        // отсортируем офисы по кол-ву людей для данного митинга в этом офисе.
        // Если людей поровну, то по id офиса (предполагаем, что чем меньше id, тем больше загруженность)
        var candidate = officeReq
                .stream()
                .sorted(reqCountCmp.thenComparing(or -> or.officeId))
                .map(or -> or.officeId)
                // roomsByOffice - список переговорок в каждом офисе, которые нам подходят
                // сложим все в один список упорядочив по офису и фитнесс функции
                .flatMap(id -> roomsByOffice.getOrDefault(id, Collections.emptyList())
                        .stream()
                        .sorted((r1, r2) -> BookingUtils.resourceComparator(officeGolden.get(id)).reversed().compare(r1.info, r2.info)))
                .findFirst();

        // если ни в одном офисе нет переговорки, то возвращаем результат без переговорок
        if (candidate.isEmpty()) {
            var interval = findInterval(allPersonTimeTableWR, duration);
            LOG.info("return empty meeting at {}", interval);
            return FullSearchResult.of(
                    SearchResult.empty(officeReq, interval)
            );
        }

        LOG.info("return partially completed meeting");
        var rwt = candidate.get();
        return FullSearchResult.of(new SearchResult(
                requiredIds,
                List.of(rwt.info),
                rwt.timeTable.freeIntervals(duration).get(0).withDurationAfterStart(duration)));

    }

    /**
     * метод аналогичен findTimeSlotAndRoom, но предназначен для регулярных встреч
     * принцип работы такой: находим возможность поставить регулярнуб встречу в будущем, затем через findTimeSlotAndRoom заполняем промежуток между текущим моментом и началом регулярки разовыми встречами
     */
    @Nonnull
    public FullSearchResult findTimeSlotAndRoomForRegular(@Nonnull TimeTable timeRestrictions,
                                                                    @Nonnull Duration duration,
                                                                    @Nonnull String organizer,
                                                                    @Nonnull Set<String> attendees,
                                                                    @Nonnull Set<String> additionalAttendees,
                                                                    @Nonnull Resource.Info roomRestrictions,
                                                                    @Nonnull Repetition repetition) {

        var allProfiles = staffClient.getByLogins(
                Stream.concat(
                        attendees.stream(),
                        additionalAttendees.stream()
                ).collect(Collectors.toSet()),
                false);

        // тут для каждого офиса создаем требования по кол-ву людей
        var officeReq = createOfficeRequirements(allProfiles);
        if (officeReq.isEmpty()) {
            return FullSearchResult.empty(FullSearchResult.MissingReason.NO_OFFICE);
        }

        // попросим саджест предложить время и место для регулярки
        // затем добьем разовыми встречами интервал до первой регулярки
        var suggested = suggest(
                staffClient.getByLogin(organizer),
                timeRestrictions.terms().getStart().toDate(),
                duration,
                repetition,
                officeReq.stream().map(or -> or.officeId).collect(Collectors.toUnmodifiableSet()),
                new ArrayList<>(attendees)
        );
        LOG.info("suggested {} options", suggested.getIntervals().size());

        var officeGolden = extendOfficeRequirements(officeReq, roomRestrictions);

        var officeIds = officeGolden.keySet();

        // проверять доступность людей - не надо, это за нас сделал саджес
        // но нужно проверить временные интервалы на соответствие требованиям от пользователя
        // так же проверим, что есть переговорки без ограничений на время бронирование (dueDate)
        var fitIntervals = suggested.getIntervals().stream()
                .filter(i -> {
                    var start = rollbackToTerm(i.getStart(), timeRestrictions.terms());
                    var end = rollbackToTerm(i.getEnd(), timeRestrictions.terms());

                    assert end.after(start);

                    return timeRestrictions.freeIntervals().stream().anyMatch(fi ->
                            fi.getStart().isBefore(start.getTime() + 1)
                                    && fi.getEnd().isAfter(end.getTime() - 1));
                })
                //
                .filter(i -> i
                        .getPlaces()
                        .stream()
                        .allMatch(place ->
                                place.getResources().stream().anyMatch(r -> r.getDueDate() == null && !"private-room".equals(r.getType())))
                )
                .collect(Collectors.toUnmodifiableList());

        // eсли нет подходящих интервалов, то выходим
        if (fitIntervals.isEmpty()) {
            LOG.info("suggest did not find any valid meeting time, fail");
            return FullSearchResult.emptyRegular(FullSearchResult.MissingReason.REGULAR_BOOKING_UNAVAILABLE);
        }

        // теперь у нас точно есть подходящий интервал времени + переговорки для регулярной встречи
        // выберем лучшую комбинацию переговорок, забронируем регулярку и добьем промежуток времени от текущего моента до начала регулярки разовыми событиями
        var result = fitIntervals.stream().map(i1 -> {
                var bestRoomsWithScore = i1.getPlaces().stream().map(p -> {
                    var golden = officeGolden.get(p.getOfficeId());
                    // берем переговорку с лучшим скором (наиболее подходящую)
                    var bestRoom = p
                            .getResources()
                            .stream()
                            .filter(r -> r.getDueDate() == null)
                            .filter(r -> !"private-room".equals(r.getType()))
                            .max(
                                    BookingUtils.resourceComparator(golden)
                            ).orElseThrow(() -> new IllegalStateException("can't find room"));
                    // возвращаем лучшую переговорку в офисе и скор
                    return Pair.of(List.of(bestRoom), BookingUtils.fitness(golden, bestRoom));
                }).reduce(
                        // редьюс должен получить список лучших переговорок в офисе и их суммарный скор
                        // начальный стейт - без переговорок и скора
                        Pair.of(Collections.emptyList(), 0.),
                        (left, right) -> Pair.of(
                                // мерджим списки переговорок
                                Stream.concat(left.getKey().stream(), right.getKey().stream()).collect(Collectors.toUnmodifiableList()),
                                // суммируем скор
                                left.getValue() + right.getValue())
                );
                // запишем -score что бы имень natural order при сортировке
                return Triple.of(i1, bestRoomsWithScore.getKey(), -bestRoomsWithScore.getValue());
            }).sorted(Comparator.comparing(Triple::getRight))
                .map(t -> {
                    assert officeIds.size() == t.getMiddle().size();
                    return new SearchResult(
                            officeIds,
                            t.getMiddle(),
                            t.getLeft().interval()
                    );
                })
                .collect(Collectors.toUnmodifiableList());

        if (result.isEmpty()) {
            return FullSearchResult.emptyRegular(FullSearchResult.MissingReason.NO_FREE_ROOMS);
        }

        var bestResult = result.get(0);

        // дополнительные встречи, пока не начались регулярные
        List<Date> additionalMeetingsDates = new ArrayList<>();

        var cDate = bestResult.interval().getStart().toDate();
        var now = timeRestrictions.terms().getStart().toDate();

        while (true) {
            cDate = getPrevRepetition(cDate, repetition);
            if (cDate.after(now)) {
                additionalMeetingsDates.add(cDate);
            } else {
                break;
            }
        }

        LOG.info("additional meeting: {} meetings are required", additionalMeetingsDates.size());

        /* для каждой даты создаем новые ограничения, основанные на дне недели
           затем пытаемся назначить встречи */
        var additionalMeetings = additionalMeetingsDates.parallelStream()
                .map(d -> extractDailyTimeTable(timeRestrictions, d))
                .map(newTimeRestriction ->
                        findTimeSlotAndRoom(newTimeRestriction, duration, organizer, attendees, additionalAttendees, roomRestrictions, true)
                )
                .filter(x -> !x.isEmpty())
                // дальше надо просто развернуть FullSearchResult в SearchResult
                .map(m -> m.additional().stream().findFirst())
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toUnmodifiableList());

        return FullSearchResult.regularOf(bestResult, repetition, additionalMeetings);
    }

    private Map<Integer, Resource.Info> extendOfficeRequirements(@Nonnull List<OfficeRequirements> officeReq, @Nonnull Resource.Info roomRestrictions) {
        return officeReq.stream().collect(Collectors.toMap(
                or -> or.officeId,
                or -> BookingUtils.withOfficeRestriction(
                        BookingUtils.withCapacityRestriction(roomRestrictions, or.count),
                        or.officeId
                ))
        );
    }

    private Response bookSingle(@Nonnull SearchResult meeting,
                                @Nonnull BasicMeetingInfo meetingInfo,
                                @Nonnull Set<String> attendees,
                                @Nullable Repetition repetition) {
        // объединяем людей и переговорки
        var invited = Stream.concat(
                attendees.stream().map(s -> s + AT_YANDEX),
                meeting.rooms().stream().map(Resource.Info::getEmail)
        ).collect(Collectors.toUnmodifiableList());


        var finalRepetition = Optional.ofNullable(repetition)
                .map(r -> {
                    var newRepetition = RepetitionUtils.buildRepetition(repetition.getType(), repetition.getEach());
                    var day = meeting.interval().getStart().dayOfWeek().getAsShortText(Locale.ENGLISH).toLowerCase();
                    newRepetition.setWeeklyDays(day);
                    return newRepetition;
                }).orElse(null);

        LOG.info("creating meeting {} | {} | {} | {}", meeting.interval(), repetition, meeting.rooms(), invited);
        var event = new WebEventCreateData(
                EventType.USER,
                meeting.interval().getStart().toDate(),
                meeting.interval().getEnd().toDate(),
                meetingInfo.meetingName(),
                meetingInfo.meetingDesc(),
                meetingInfo.organizer() + AT_YANDEX,
                invited,
                finalRepetition
        );
        event.setParticipantsCanEdit(meetingInfo.participantsCanEdit());
        event.setParticipantsCanInvite(meetingInfo.participantsCanInvite());
        event.setOthersCanView(meetingInfo.othersCanView());

        return calendarUpdate.createEvent(event);
    }

    /**
     * фильтрует список возможных бронирований с учетом доступности для окнкретного пользоателя
     */
    private List<SearchResult> applyPersonalFilter(List<SearchResult> result, String organizer) {
        if (result.isEmpty()) {
            return Collections.emptyList();
        }

        var person = staffClient.getByLogin(organizer);
        // в качетсве интервала для поиска возьмем начало самого раннего кандидата и конец самого позднего
        var times = result.stream().map(SearchResult::interval).flatMap(i -> Stream.of(i.getStartMillis(), i.getEndMillis())).collect(Collectors.toList());

        var from = Collections.min(times);
        var to = Collections.max(times);

        var offices = result.stream().flatMap(sr -> sr.offices().stream()).collect(Collectors.toSet());

        var fullSchedule = roomService.fullScheduleForPerson(person, new Interval(from, to), offices);
        return result.stream().filter(sr -> sr.rooms().stream().allMatch(room -> {
            var roomTimeTable = fullSchedule.get(room.getEmail());
            if (roomTimeTable == null) {
                return false;
            }

            // хотя бы один свободный слот должен включать в себя потенциальное время встречи
            return roomTimeTable.freeIntervals().stream().anyMatch(i -> i.contains(sr.interval()));
        })).collect(Collectors.toUnmodifiableList());

    }

    /**
     * Берем из расписания день недели, соответствующий day и переносим все ограничения на новое расписание, составленное на дату day
     */
    private TimeTable extractDailyTimeTable(TimeTable tt, Date day) {
        final Calendar c = Calendar.getInstance();
        c.setTime(day);
        final var expectedDay = c.get(Calendar.DAY_OF_WEEK);

        assert tt.terms().toDuration().getStandardDays() < 8;
        // берем интервалы, занятости соответствующего дня и переносим их на дату day
        final var busy = tt.busyIntervals()
                .stream()
                .filter(i -> {
                    var tmp = Calendar.getInstance();
                    tmp.setTime(i.getStart().toDate());
                    return expectedDay == tmp.get(Calendar.DAY_OF_WEEK);
                })
                .map(i -> {
                    final Calendar cal = Calendar.getInstance();
                    cal.setTime(i.getStart().toDate());
                    cal.set(Calendar.YEAR, c.get(Calendar.YEAR));
                    cal.set(Calendar.MONTH, c.get(Calendar.MONTH));
                    cal.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));

                    var d = cal.getTime();
                    return new Intervalable() {
                        @Override
                        public Date getStart() {
                            return d;
                        }

                        @Override
                        public Date getEnd() {
                            return new Date(d.getTime() + i.toDurationMillis());
                        }
                    };
                })
                .collect(Collectors.toUnmodifiableList());
        return TimeTableImpl.fromEventDates(dayTerms(day), busy);
    }

    private Interval dayTerms(Date day) {
        final Calendar cal = Calendar.getInstance();
        cal.setTime(day);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);

        var start = cal.getTime();
        return new Interval(
                start.getTime(),
                start.getTime() + Duration.standardDays(1).getMillis()
        );
    }

    /**
     * метод отматывает время назад по неделям, пока не упрется в ограничения terms
     * Нужен для того, что бы масштабировать ограничения, заданные неделей на более длительные интервалы
     * @return
     */
    private Date rollbackToTerm(Date d, Interval terms) {
        if (d.getTime() < terms.getStartMillis()) {
            throw new IllegalArgumentException("date can't be before terms");
        }
        var cDate = d;
        // будем вычитать по 7 дней пока дата после конца интервала
        while (cDate.after(terms.getEnd().toDate())) {
            cDate = new Date(cDate.getTime() - Duration.standardDays(7).getMillis());
        }

        return cDate;
    }

    private SuggestResponse suggest(
            @Nonnull Person organizer,
            @Nonnull Date start,
            @Nonnull Duration duration,
            @Nonnull Repetition repetition,
            @Nonnull Set<Integer> offices,
            @Nonnull Collection<String> users) {
        var src = List.of("");
        if ("".equals(repetition.getWeeklyDays())) {
            src = List.of("mon", "tue", "wed", "thu", "fri");
        }
        return src.parallelStream().map(day -> {
            var r = RepetitionUtils.buildRepetition(repetition.getType(), repetition.getEach());
            r.setWeeklyDays(day);
            r.setDueDate(repetition.getDueDate());
            r.setMonthlyLastweek(repetition.getMonthlyLastweek());

            var body = new SuggestBody();
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(start);
            calendar.set(Calendar.HOUR_OF_DAY, 12);

            body.setOffices(offices.stream().map(SuggestBody.OfficeInfo::new).collect(Collectors.toUnmodifiableList()));
            body.setRepetition(r);
            body.setSearchStart(calendar.getTime());
            body.setEventStart(calendar.getTime());
            body.setEventEnd(new Date(calendar.getTime().getTime() + duration.getMillis()));
            body.setUsers(users.stream().map(u -> u.contains("@") ? u : u + "@yandex-team.ru").collect(Collectors.toUnmodifiableList()));
            return calendarWeb.suggestMeetingsTimes(organizer.getUid(), body);
        }).reduce(new SuggestResponse(), (r1, r2) -> {
            var r = new SuggestResponse();
            r.setIntervals(Stream.of(r1.getIntervals(), r2.getIntervals()).flatMap(Collection::stream).collect(Collectors.toUnmodifiableList()));
            r.setNextSearchStart(Optional.ofNullable(r2.getNextSearchStart()).orElse(r1.getNextSearchStart()));
            return r;
        });

    }

    private static Date getPrevRepetition(Date start, Repetition repetition) {
        Calendar c = Calendar.getInstance();
        c.setTime(start);
        if (repetition.getType() == Repetition.Type.DAILY) {
            c.add(Calendar.DAY_OF_MONTH, -repetition.getEach());
        } else if (repetition.getType() == Repetition.Type.WEEKLY) {
            c.add(Calendar.DAY_OF_MONTH, -7 * repetition.getEach());
        } else if (repetition.getType() == Repetition.Type.MONTHLY_NUMBER) {
            c.add(Calendar.MONTH, -repetition.getEach());
        } else if (repetition.getType() == Repetition.Type.YEARLY) {
            c.add(Calendar.YEAR, -repetition.getEach());
        } else {
            throw new IllegalArgumentException("unsupported repetition " + repetition.getType());
        }
        Date result = c.getTime();
        c.set(Calendar.MILLISECOND, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.HOUR, 0);
        return result;

    }

    private Interval findInterval(TimeTableImpl timeTable, Duration duration) {
        return timeTable.freeIntervals(duration).get(0).withDurationAfterStart(duration);
    }

    @Nonnull
    private List<OfficeRequirements> createOfficeRequirements(@Nonnull Collection<Person> attendees) {
        return attendees
                .stream()
                .filter(p -> p.getLocation().getOffice().getIdAsCalendarOffice() >= 0)
                .collect(Collectors.groupingBy(p -> p.getLocation().getOffice().getIdAsCalendarOffice(), Collectors.counting()))
                .entrySet()
                .stream()
                .map(e -> OfficeRequirements.with(e.getKey(), e.getValue().intValue()))
                .filter(req -> roomService.allOfficeIds().contains(req.officeId))
                .collect(Collectors.toUnmodifiableList());
    }

    private void logSearchResults(List<SearchResult> results, int count) {
        results.stream().limit(count).forEach(r -> {
            var rooms = r.rooms()
                    .stream()
                    .map(room -> room.getId() + "|" + room.getCityName() + "|" + room.getName())
                    .collect(Collectors.joining(","));
            LOG.info("example: {} at [{}]", r.interval(), rooms);
        });
    }

}
