package ru.yandex.qe.mail.meetings.booking.impl;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.inject.Inject;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.booking.PersonService;
import ru.yandex.qe.mail.meetings.booking.TimeTable;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Availability;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Decision;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;


@Service("personService")
public final class PersonServiceImpl implements PersonService {
    private static final Logger LOG = LoggerFactory.getLogger(ru.yandex.qe.mail.meetings.booking.PersonService.class);

    // SimpleDateFormat is not thread safe
    private final ThreadLocal<DateFormat> _DF = ThreadLocal.withInitial(() -> new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXXX"));

    @Nonnull
    private final CalendarWeb calendarWeb;

    @Inject
    public PersonServiceImpl(@Nonnull CalendarWeb calendarWeb) {
        this.calendarWeb = calendarWeb;
    }

    @Nonnull
    public TimeTable timetableForUid(@Nonnull String uid, @Nonnull Date from, @Nonnull Date to) {
        List<Event> events;
        try {
            events = calendarWeb.getEvents(uid, _DF.get().format(from), _DF.get().format(to), true).getEvents();
        } catch (Exception e) {
            // TODO WTF? refactor it!
            LOG.warn("sleep for retry");
            try {
                Thread.sleep(500 + ThreadLocalRandom.current().nextInt(100));
            } catch (InterruptedException ex) {
                throw new RuntimeException();
            }
            events = calendarWeb.getEvents(uid, _DF.get().format(from), _DF.get().format(to), true).getEvents();
        }

        events = events
                    .stream()
                    .filter(e -> e.getDecision() != null && e.getDecision() != Decision.NO)
                    .filter(e -> e.getWebEventUserData().getAvailability() != Availability.AVAILABLE)
                    .filter(e -> {
                        var layerId = e.getWebEventUserData().getLayerId();
                        var layer = calendarWeb.getLayer(uid, layerId);
                        return layer.isAffectAvailability();
                    })
                    .collect(Collectors.toList());
        var terms = new Interval(from.getTime(), to.getTime());
        return TimeTableImpl.fromEventDates(terms, events);
    }

    @Nonnull
    public Map<Person, TimeTable> timetableForPersonsParallel(@Nonnull List<Person> persons, @Nonnull Date from, @Nonnull Date to) {
        return persons
                .parallelStream()
                .peek(p -> LOG.info("getting timetable for {}/{}", p.getLogin(), p.getUid()))
                .map(p -> Pair.of(p, timetableForUid(p.getUid(), from, to)))
                .collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }
}
