package ru.yandex.qe.mail.meetings.booking.impl;


import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;

import org.joda.time.Interval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.booking.RoomService;
import ru.yandex.qe.mail.meetings.booking.TimeTable;
import ru.yandex.qe.mail.meetings.booking.util.BookingUtils;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Intervalable;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Office;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Offices;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.DateRange;

@Service("roomService")
public final class RoomServiceImpl implements RoomService {
    private static final Logger LOG = LoggerFactory.getLogger(RoomServiceImpl.class);

    @Nonnull
    private final CalendarWeb calendarWeb;

    // кол-во дней в кеше расписании
    private final Duration cacheSize;

    private final boolean isEnabled;

    private volatile CacheState cacheState;

    private final ConcurrentMap<String, TimeTable> roomsSchedule = new ConcurrentHashMap<>();

    private final ReentrantLock lock = new ReentrantLock();

    @Inject
    public RoomServiceImpl(@Nonnull CalendarWeb calendarWeb,
                           @Nonnull @Value("${rooms.cache.days:60}") Long cacheSizeDays, @Value("${booking" +
            ".enabled:false}") boolean isEnabled) {
        this.calendarWeb = calendarWeb;
        this.cacheSize = Duration.ofDays(cacheSizeDays);
        this.isEnabled = isEnabled;
    }

    public Set<Integer> allOfficeIds() {
        ensureInitialized();
        return new TreeSet<>(this.cacheState.roomByOffice.keySet());
    }

    /**
     * получает список ресурсов "не хуже", чем указанный
     */
    @Nonnull
    public List<Resource.Info> roomsWithRestrictions(@Nonnull Resource.Info golden) {
        ensureInitialized();
        return cacheState.roomCache
                .values()
                .stream()
                .filter(
                        BookingUtils.filter(golden)
                ).collect(
                        Collectors.toUnmodifiableList()
                );
    }

    @Override
    public Optional<Resource.Info> byMail(@Nonnull String mail) {
        ensureInitialized();
        return Optional.ofNullable(cacheState.roomCache.get(mail));
    }

    /**
     * аналогично @see `roomsWithRestrictions`, но дополнительно ресурсы сортируются согласно BookintgUtils::fitness
     */
    @Nonnull
    public List<Resource.Info> roomsWithRestrictionsAndOrder(@Nonnull Resource.Info golden) {
        return roomsWithRestrictions(golden)
                .stream()
                .sorted(
                        BookingUtils.resourceComparator(golden).reversed()
                )
                .collect(
                        Collectors.toUnmodifiableList()
                );
    }

    public ResourceWithTimeTable schedule(@Nonnull Resource.Info room, @Nonnull Interval terms) {
        var factor = 1000 * 60 * 5;
        var ts = (System.currentTimeMillis() / factor) * factor;

        if (terms.getEndMillis() < ts) {
            throw new IllegalStateException("can't search in past");
        }

        if (terms.getStartMillis() > ts) {
            terms = new Interval(ts, terms.getEndMillis());
        }

        assert roomsSchedule.containsKey(room.getEmail());
        var roomTimeTable = roomsSchedule.get(room.getEmail());
        if (roomTimeTable == null) {
            LOG.warn("roomTimeTable is null for {}, roomsSchedule: [{}]", room, roomsSchedule.toString());
        }
        var roomTerms = roomTimeTable.terms();

        if (!BookingUtils.fuzzyIntervalContains(roomTerms, terms, Duration.ofMinutes(1))) {
            // интервалы не совпадают, придется перезагрузить расписание
            // пока кинем эксепшен
            throw new IllegalArgumentException("Timetable is unavailable now");
        }

        return new ResourceWithTimeTable(room, TimeTableImpl.fromEventDates(terms, roomTimeTable.busyIntervalsEx()));

    }

    @Override
    public Map<String, TimeTable> fullScheduleForPerson(@Nonnull Person person, @Nonnull Interval terms) {
        return fullScheduleForPerson(person, terms, allOfficeIds());
    }

    public Map<String, TimeTable> fullScheduleForPerson(@Nonnull Person person, @Nonnull Interval terms,
                                                        @Nonnull Set<Integer> offices) {
        var factor = 1000 * 60 * 5;
        var ts = (System.currentTimeMillis() / factor) * factor;

        if (terms.getEndMillis() < ts) {
            throw new IllegalStateException("can't search in past");
        }

        if (terms.getStartMillis() > ts) {
            terms = new Interval(ts, terms.getEndMillis());
        }

        final var fterms = terms;
        return officeSchedule(offices, terms.getStart().toDate(), terms.getEnd().toDate(), person.getUid())
                .stream()
                .flatMap(o -> o
                        .getResources()
                        .stream()
                        .filter(r -> cacheState.roomCache.containsKey(r.getResourceInfo().getEmail()))
                ).collect(Collectors.toMap(r -> r.getResourceInfo().getEmail(),
                        r -> TimeTableImpl.fromEventDates(fterms, eventsWithRestrictions(r, fterms))));
    }

    /**
     * Для некоторых переговорок вводятся ограничения на бронирования, здесь они учитываются как интервалы занятости
     *
     * @param resource - ресурс
     * @param terms    - ограничения по времени
     * @return - объединенный список событий и фейковых событий - ограничений
     */
    @Nonnull
    private List<? extends Intervalable> eventsWithRestrictions(Resource resource, Interval terms) {
        var restrictions = resource.getRestrictions()
                .stream()
                .map(restriction -> {
                    var start = restriction.getStart();
                    var end = restriction.getEnd() != null ? restriction.getEnd() : terms.getEnd().toDate();

                    if (start.getTime() > end.getTime()) {
                        return null;
                    }

                    return new Intervalable() {
                        @Override
                        public Date getStart() {
                            return start;
                        }

                        @Override
                        public Date getEnd() {
                            return end;
                        }
                    };
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toUnmodifiableList());

        return Stream.concat(restrictions.stream(), resource.getEvents().stream()).collect(Collectors.toUnmodifiableList());
    }

    @Scheduled(fixedDelay = 300000L, initialDelayString = "${room.services.cron.initial.delay:0}")
    public void updateRoomInfo() {
        if (!isEnabled) {
            LOG.debug("booking is disabled, skipping room info update");
            return;
        }

        lock.lock();
        try {
            if (Optional.ofNullable(cacheState)
                    // если стейт есть, берем сколько времени назад он был сделан
                    .map(cs -> System.currentTimeMillis() - cs.createTs)
                    // если прошло меньше 10 секунд, то стейт еще ок
                    .filter(d -> d > Duration.ofSeconds(10).toMillis())
                    // есть актуальный стейт?
                    .isPresent()) {
                return;
            }
            var tmpRoomByOffice = new HashMap<Integer, List<Resource.Info>>();
            var tmpRoomCache = new HashMap<String, Resource.Info>();

            // список айдишников офисов
            var allOfficesIds = calendarWeb
                    .getOffices()
                    .getOffices()
                    .stream()
                    .map(Office::getId)
                    .collect(Collectors.toList());


            // Запрашиваем расписание всех офисов с сегодняшнего дня (00:00) до сегодня + $rooms.cache.days.
            var today = DateRange.today();

            calendarWeb
                    .getResourceSchedule(allOfficesIds, today,
                            new Date(today.getTime() + Duration.ofDays(1).toMillis()))
                    .getOffices()
                    .forEach(o -> {
                        var rooms = o.getResources().stream()

                                // только переговорки
                                // тут небольшие проблемы: r.getResourceInfo().getResourceType() == null (!), поэтому
                                // сравниваем "room"
                                .filter(r -> "room".equals(r.getResourceInfo().getType()))

                                // сейчас не хватает полей (например, вметсимости), нужно взять email и отдельно
                                // получить описание
                                // по id переговорки получить ее описнаие нельзя :(
                                .map(r -> r.getResourceInfo().getEmail())
                                .map(calendarWeb::getResourceDescription)
                                .collect(Collectors.toList());

                        rooms.forEach(r -> {
                            tmpRoomCache.put(r.getEmail(), r);
                            tmpRoomByOffice
                                    .computeIfAbsent(o.getId(), __ -> new ArrayList<>())
                                    .add(r);
                        });
                    });

            this.cacheState = new CacheState(tmpRoomCache, tmpRoomByOffice);
        } finally {
            lock.unlock();
        }
    }

    @Scheduled(fixedDelay = 60000L, initialDelayString = "${room.services.cron.initial.delay:0}")
    public void updateSchedule() {
        if (!isEnabled) {
            LOG.debug("booking is disabled, skipping schedule update");
            return;
        }

        initializeIfNeeded();

        var today = DateRange.today();

        // накидываем день, т.к. нужна граница справа, а апи возвращает включительно
        var cacheTerms = new Interval(today.getTime(),
                today.getTime() + this.cacheSize.toMillis() + Duration.ofDays(1).toMillis());

        // получаем расписание переговорок на `cacheSize` времени вперед.
        // Фильтруем только известные ресурсы (переговорки) и составляем расписание
        officeSchedule(allOfficeIds(), today, new Date(today.getTime() + this.cacheSize.toMillis()))
                .stream()
                .flatMap(o -> o
                        .getResources()
                        .stream()
                        .filter(r -> cacheState.roomCache.containsKey(r.getResourceInfo().getEmail()))
                ).forEach(r -> {
            var timeTable = TimeTableImpl.fromEventDates(cacheTerms, r.getEvents());
            roomsSchedule.put(r.getResourceInfo().getEmail(), timeTable);
        });
    }

    private void initializeIfNeeded() {
        if (this.cacheState != null) {
            return;
        }
        LOG.info("manual room initialization");
        updateRoomInfo();
    }

    private List<Office> officeSchedule(Set<Integer> officeIds, Date from, Date to) {
        return officeSchedule(officeIds, from, to, null);
    }

    private List<Office> officeSchedule(Set<Integer> officeIds, Date from, Date to, @Nullable String uid) {
        assert from.getTime() < to.getTime();

        // Бекенд делает sql-запрос, в котором перечисляет все встречи, которые надо вернуть.
        // Например: SQL [SELECT event_id, start_ts FROM rdate WHERE (is_rdate = ?) AND (event_id IN (?{100K times}))
        // AND (start_ts <= ?) AND (start_ts >= ?)];
        // Что естественно приводит к ощибке. Поэтому лучше запрашивать инфу по офисам отдельно
        return officeIds.stream().parallel().flatMap(id ->
        {
            Offices resourceSchedule = null;
            try {
                resourceSchedule = calendarWeb.getResourceSchedule(List.of(id), from, to, uid);
            } catch (Exception e) {
                LOG.error("error {} on officeIds = {}, from = {}, to = {}, uid = {}",
                        e.getMessage(), id, from, to, uid);
                throw e;
            }
            return resourceSchedule.getOffices().stream();
        }).collect(Collectors.toUnmodifiableList());
    }

    private void ensureInitialized() {
        if (this.cacheState == null) {
            for (int i = 0; lock.isLocked() || i > 10; i++) {
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException e) {
                    //nothing
                }
            }
            if (this.cacheState == null) {
                throw new IllegalStateException("class " + this.getClass().getSimpleName() + "::cacheState is not " +
                        "initilized yet");
            }
        }
    }

    private static class CacheState {
        final Map<String, Resource.Info> roomCache;
        final Map<Integer, List<Resource.Info>> roomByOffice;
        final long createTs = System.currentTimeMillis();

        CacheState(Map<String, Resource.Info> roomCache, Map<Integer, List<Resource.Info>> roomByOffice) {
            this.roomCache = Collections.unmodifiableMap(roomCache);
            this.roomByOffice = Collections.unmodifiableMap(roomByOffice);
        }
    }
}
