package ru.yandex.qe.mail.meetings.booking.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Interval;

import ru.yandex.qe.mail.meetings.booking.TimeTable;
import ru.yandex.qe.mail.meetings.booking.util.IntervalUtils;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Intervalable;

public class TimeTableImpl implements TimeTable {
    private static final long SPLIT_INTERVAL = Duration.standardMinutes(5).getMillis();

    private final List<? extends Intervalable> busy;
    private final Interval terms;
    private final List<Interval> freeIntervals;

    public static TimeTableImpl fromEventDates(@Nonnull Interval terms, @Nonnull List<? extends Intervalable> eventDates) {
        return new TimeTableImpl(terms, eventDates);
    }


    @Nonnull
    public static TimeTableImpl intersectFree(@Nonnull Interval terms, @Nonnull TimeTable... timeTables) {
        return intersectFree(terms, Arrays.asList(timeTables));
    }

    public static TimeTableImpl intersectFree(@Nonnull Interval terms, @Nonnull Collection<TimeTable> timeTables) {
        return TimeTableImpl.fromEventDates(
                terms,
                timeTables.stream()
                        .flatMap(t ->
                                IntervalUtils.makeBusy(terms, t.freeIntervals()).stream())
                        .map(i -> new Intervalable() {
                            @Override
                            public Date getStart() {
                                return i.getStart().toDate();
                            }

                            @Override
                            public Date getEnd() {
                                return i.getEnd().toDate();
                            }
                        }).collect(Collectors.toUnmodifiableList()));
    }

    @Nonnull
    public static TimeTableImpl intersectFair(@Nonnull Interval terms, @Nonnull TimeTable... timeTables) {
        return intersectFair(terms, Arrays.asList(timeTables));
    }

    public static TimeTableImpl intersectFair(@Nonnull Interval terms, @Nonnull Collection<TimeTable> timeTables) {
        // если будет тормозить, то надо просто сохранить в конструкторе mask, выравнивать их и мерджить.
        // пока попробуем в лоб
        return TimeTableImpl.fromEventDates(
                terms,
                timeTables
                        .stream()
                        .flatMap(t -> t.busyIntervalsEx().stream()).collect(Collectors.toUnmodifiableList())
        );
    }

    @Override
    public Interval terms() {
        return terms;
    }

    @Override
    @Nonnull
    public List<Interval> freeIntervals(@Nonnull Duration minDuration) {
        return freeIntervals
                .stream()
                // не используем isLonger, т.к. хотим >=
                .filter(i -> !i.toDuration().isShorterThan(minDuration))
                .collect(Collectors.toUnmodifiableList());
    }

    @Override
    @Nonnull
    public List<? extends Intervalable> busyIntervalsEx() {
        return Collections.unmodifiableList(busy);
    }

    private TimeTableImpl(@Nonnull Interval terms, @Nonnull List<? extends Intervalable> eventDates) {
        this.busy = eventDates;
        this.terms = terms;

        // хотим построить интервалы, когда пользователь ничем не занят
        // знаем, что есть перекрывающиеся встречи, что немного затрудняет жизнь,
        // но так же знаем, что всчтречи выравнены на 15 минут (это не всегда правдя, поэтому выравниваем на 1 минуту)

        // массив, где каждому 1-минутному интервалу соответствует флаг занятости

        var mask = split15(terms);
        var s = terms.getStartMillis();
        eventDates.forEach(e -> {
            for (int i = indexOf(s, e.getStart().getTime()); i < indexOf(s, e.getEnd().getTime()); i++) {
                // чисто в теории могут буть отрицательные индексы, если встреча переходит за полночь, а расписание составляется с 00:00. т.е.
                // встреча стартует как бы за n минут до рассматриваемого периода
                if (i >= 0 && i < mask.length) {
                    mask[i] = true;
                }
            }
        });

        // теперь нужно вычленить свободные интервалы
        this.freeIntervals = extractSeq(mask, false)
                .stream()
                // тут нам возвращаются индексы свободных пятнадцатиминуток. Что бы получить правую границу - увеличиваем конечный индекс на 1
                .map(p ->new Interval(s + SPLIT_INTERVAL * p.getLeft(), s + SPLIT_INTERVAL * (1 + p.getRight())))
                .collect(Collectors.toList());


    }

    private List<Pair<Integer, Integer>> extractSeq(boolean[] mask, boolean target) {
        var result = new ArrayList<Pair<Integer, Integer>>();
        int start = 0;

        while (start < mask.length) {
            start = findNext(mask, target, start);
            if (start < 0) {
                break;
            }
            var nextUnfit = findNext(mask, !target, start + 1);
            if (nextUnfit < 0) {
                nextUnfit = mask.length;
            }
            var end = nextUnfit - 1;

            if (start != end) {
                result.add(Pair.of(start, end));
            }
            start = nextUnfit;
        }

        return result;

    }

    private void ensureAligned(DateTime start) {
        if (Math.abs(SPLIT_INTERVAL * (start.getMillis() / SPLIT_INTERVAL) - start.getMillis()) > 1000L) {
//            throw new IllegalArgumentException("argument must be aligned on 15 min (" + start + ")");
        }
    }

    private int indexOf(long start, long x) {
        int idx = (int)Math.round((x - start) / (1. * SPLIT_INTERVAL));
        //assert x >= start;
        assert Math.abs(x - (start + idx * SPLIT_INTERVAL)) < 1000L;

        return idx;
    }

    private boolean[] split15(Interval interval) {
        ensureAligned(interval.getEnd());
        ensureAligned(interval.getStart());
        var len = indexOf(terms.getStartMillis(), terms.getEndMillis());
        return new boolean[len];
    }

    private int findNext(boolean[] a, boolean v, int offset) {
        for (int i = offset; i < a.length; i++) {
            if (a[i] == v) {
                return i;
            }
        }
        return -1;
    }

    @Override
    public String toString() {
        return "TimeTableImpl{" +
                "busy=" + busy +
                ", terms=" + terms +
                ", freeIntervals=" + freeIntervals +
                '}';
    }
}
