package ru.yandex.qe.mail.meetings.booking.util;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Interval;

import ru.yandex.qe.mail.meetings.booking.RoomService;
import ru.yandex.qe.mail.meetings.booking.TimeTable;
import ru.yandex.qe.mail.meetings.booking.impl.TimeTableImpl;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Intervalable;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;


public enum  BookingUtils {
    ;

    /**
     * Возвращает фильтр, который отбирает ресурсы "не хуже", чем golden
     */
    @Nonnull
    public static Predicate<Resource.Info> filter(@Nonnull Resource.Info golden) {
        return r -> {
            boolean missmatch = false;
            // важные проверки: офис, вместительности
            // по решению alena@ не можно бронировать переговорки, не удовлетворяющие доп. условиям, например, конференц связь
            // если понадобится это изменить, то можно добавить сюда еще условий
            missmatch |= r.getOfficeId() != golden.getOfficeId();
            missmatch |= r.getCapacity() < golden.getCapacity();
            return !missmatch;
        };
    }

    @Nonnull
    public static Comparator<Resource.Info> resourceComparator(@Nonnull Resource.Info golden) {
        return Comparator.comparingDouble(x -> fitness(golden, x));
    }


    /**
     * Вычисляет fitness-функцию между golden и текущим ресурсом;
     * Чем больше, тем более подходящие ресрсы.
     * предполагается, что переговорки совместимы в смысле методв `filter`
     * Сюда можено заложить всякие эвристики удобства, например:
     *  - на троих больше подходят маленькие переговорки
     *  - если связь не нужна6 то лучше брать переговорки без телефона
     *  - ...
     */
    public static double fitness(@Nonnull Resource.Info golden, @Nonnull Resource.Info x) {
        // -0.25 очков за каждое лишнее кресло
        var score = (golden.getCapacity() - x.getCapacity()) / 4.;

        score += featureImportance(golden, x, Resource.Info::isHasPhone, -10, -0.1);
        score += featureImportance(golden, x, Resource.Info::isVoiceConferencing, -10, -0.1);
        score += featureImportance(golden, x, r -> n2f(r.isHasVideo()) || (r.getVideo() != null && !"".equals(r.getVideo())), -10, -0.2);

        score += featureImportance(golden, x, Resource.Info::isMarkerBoard, -5, -0.05);
        score += featureImportance(golden, x, Resource.Info::isGuestWifi, -5,-0.05);
        score += featureImportance(golden, x, r -> r.getLcdPanel() > 0, -5, -0.05);
        score += featureImportance(golden, x, r -> r.getProjector() > 0, -5, -0.05);

        // сюда же можно впихнуть логику про расстояния от предыдущей переговорки, lcd-панели и тд

        return score;
    }

    /**
     * добавить огнаричения на локацию офиса к переговорке
     * @param roomRestrictions - текцщие ограничения
     * @param officeId - желаемый офис
     * @return - новые ограничения
     */
    public static Resource.Info withOfficeRestriction(@Nonnull Resource.Info roomRestrictions, int officeId) {
        var cpy = roomRestrictions.cloneRestrictions();
        cpy.setOfficeId(officeId);
        return cpy;
    }

    /**
     * Добавить огранияения на вместимость переговорки
     */
    public static Resource.Info withCapacityRestriction(@Nonnull Resource.Info roomRestrictions, int capacity) {
        var cpy = roomRestrictions.cloneRestrictions();
        cpy.setCapacity(capacity);
        return cpy;
    }

    /**
     * Находится ли один интервал внутри другого с допустимой погрешностью
     */
    public static boolean fuzzyIntervalContains(@Nonnull Interval outter, @Nonnull Interval inner, @Nonnull Duration e) {
        return
                (
                        outter.getStartMillis() <= inner.getStartMillis()
                                ||  outter.getStartMillis() - inner.getStartMillis() < e.toMillis()
                )
                && (
                        outter.getEndMillis() >= inner.getEndMillis()
                                ||  inner.getEndMillis() - outter.getEndMillis() < e.toMillis()
                )
                ;
    }

    /**
     * @param officeGolden - словарь офис -> идеальная переговорка
     * @param roomByOffice - словарь офис -> список переговорок и их расписания
     * @param terms - временные ограничения
     * @return - упорядоченный по приоритету список переговорок и времен встреч
     */
    public static List<Pair<List<Resource.Info>, TimeTable>> combineAllAndOrder(
            @Nonnull Map<Integer, Resource.Info> officeGolden,
            @Nonnull Map<Integer, List<RoomService.ResourceWithTimeTable>> roomByOffice,
            @Nonnull Interval terms) {

        if (roomByOffice.isEmpty() || roomByOffice.values().stream().mapToInt(List::size).sum() == 0) {
            return Collections.emptyList();
        }

        var allowedIds = new TreeSet<>(roomByOffice.keySet());

        // компаратор сравнивает пары ([переговорки], расписание) в следующем порядке:
        // 1) сравниваем сумму фитнесс функций от списка переговорок
        // 2) выбираем те пары, которые обеспецивают более раннюю встречу
        Comparator<Pair<List<Resource.Info>, TimeTable>> comparator = (p1, p2) -> {
            Comparator<List<Resource.Info>> inner = Comparator.comparing(
                    l -> l.stream().mapToDouble(x -> -fitness(officeGolden.get(x.getOfficeId()), x)).sum()
            );


            var r = inner.compare(p1.getKey(), p2.getKey());
            if (r == 0) {
                r = Long.compare(p1.getValue().freeIntervals().get(0).getStartMillis(), p2.getValue().freeIntervals().get(0).getStartMillis());
            }
            return r;
        };

        // тут просто сортируем, основная работа в `combineAllInner`
        return combineAllInner(
                roomByOffice,
                allowedIds,
                TimeTableImpl.fromEventDates(terms, Collections.emptyList()),
                Collections.emptyList()
        )
                .stream()
                .sorted(comparator)
                .collect(Collectors.toUnmodifiableList());
    }

    @Nonnull
    private static List<Pair<List<Resource.Info>, TimeTable>> combineAllInner(
            @Nonnull Map<Integer, List<RoomService.ResourceWithTimeTable>> roomByOffice,
            SortedSet<Integer> allowedOffices,
            TimeTable currentTimeTable,
            List<Resource.Info> state) {

        // хотим получить все комбинации переговорок (и время), в которых можно провести встречу

        // конец рекурсии, офисов больше нет
        if (allowedOffices.isEmpty()) {
            return List.of(Pair.of(
                    state,
                    currentTimeTable
            ));
        }

        // аккумулятор результата
        List<Pair<List<Resource.Info>, TimeTable>> result = new ArrayList<>();

        // из какого офиса будем брать переговорки на этой глубине рекурсии
        var officeId = allowedOffices.first();

        // из каких офисов надо будет еще набрать переговорок
        var newAllowedIds = new TreeSet<>(allowedOffices);
        newAllowedIds.remove(officeId);

        // переговорки в текущем офисе
        var rwtList = roomByOffice.get(officeId);

        for (var rwt : rwtList) {
            // перебираем все переговорки текущего офиса и добавляем каждую к текущему стейту (переговоркам которые были добавлены выше в рекурсии)
            var newState = new ArrayList<>(state);
            newState.add(rwt.info);

            //теперь надо проверить валидность стейта: должны быть слоты под встречу
            // делаем пересечение свободных слотов у совместного расписания переговорок из стейта и текущей переговорки
            var newTimeTable = TimeTableImpl.intersectFree(currentTimeTable.terms(), currentTimeTable, rwt.timeTable);

            if (!newTimeTable.freeIntervals().isEmpty()) {
                // если есть свободные слоты, значит мы можем поставить встречу в данный набор переговорок
                // осталось только расширить текущий набор переговорками из остальных офисов (newAllowedIds)
                result.addAll(
                        combineAllInner(roomByOffice, newAllowedIds, newTimeTable, newState)
                );
            }
        }

        return result;
    }

    public static Interval termsFromData(@Nonnull List<? extends Intervalable> data) {
        var times = data.stream().flatMap(d -> Stream.of(d.getStart().getTime(), d.getEnd().getTime())).collect(Collectors.toList());
        var from = Collections.min(times);
        var to = Collections.max(times);
        return new Interval(from, to);
    }

    public static Intervalable intervalable(@Nonnull Interval interval) {
        return new Intervalable() {
            @Override
            public Date getStart() {
                return interval.getStart().toDate();
            }

            @Override
            public Date getEnd() {
                return interval.getEnd().toDate();
            }
        };
    }

    /**
     * вычисляет штраф за наличии/отсутствие фичи переговорки
     * @param golden - идеальная переговорка
     * @param current - текущая переговорка
     * @param trans - лямбда для проверки наличия какой-либо фичи
     * @param fp - штраф за отсутствие фичи
     * @param fn - штраф за наличие фичи
     * @return - штраф
     */
    private static double featureImportance(Resource.Info golden, Resource.Info current, Function<Resource.Info, Boolean> trans, double fp, double fn) {
        var gt = n2f(trans.apply(golden));
        var rt = n2f(trans.apply(current));

        // требование не выполняется
        if (gt && !rt) {
            return fp;
        }

        // требования нет, но оно удовлетворено
        if (!gt && rt) {
            return fn;
        }
        return 0.;
    }

    private static boolean n2f(Boolean b) {
        return b == null ? false : b;
    }
}


