package ru.yandex.qe.mail.meetings.booking.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.joda.time.Interval;

public final class IntervalUtils {
    public static class AnnotatedPoint implements Comparable<AnnotatedPoint> {
        public long value;
        public PointType type;

        public AnnotatedPoint(long value, PointType type) {
            this.value = value;
            this.type = type;
        }

        @Override
        public int compareTo(AnnotatedPoint other) {
            if (other.value == this.value) {
                return this.type.ordinal() < other.type.ordinal() ? -1 : 1;
            } else {
                return this.value < other.value ? -1 : 1;
            }
        }

        // the order is important here: if multiple events happen at the same point,
        // this is the order in which you want to deal with them
        public enum PointType {
            End, GapEnd, GapStart, Start
        }
    }

    public static List<Interval> makeBusy(Interval term, List<Interval> freeIntervals) {
        var queue = initQueue(List.of(term), freeIntervals);
        var result = doSweep(queue);
        return result;
    }

    private static List<AnnotatedPoint> initQueue(List<Interval> interval, List<Interval> remove) {
        // annotate all points and put them in a list
        List<AnnotatedPoint> queue = new ArrayList<>();
        for (Interval i : interval) {
            queue.add(new AnnotatedPoint(i.getStartMillis(), AnnotatedPoint.PointType.Start));
            queue.add(new AnnotatedPoint(i.getEndMillis(), AnnotatedPoint.PointType.End));
        }
        for (Interval i : remove) {
            queue.add(new AnnotatedPoint(i.getStartMillis(), AnnotatedPoint.PointType.GapStart));
            queue.add(new AnnotatedPoint(i.getEndMillis(), AnnotatedPoint.PointType.GapEnd));
        }

        // sort the queue
        Collections.sort(queue);

        return queue;
    }

    private static List<Interval> doSweep(List<AnnotatedPoint> queue) {
        List<Interval> result = new ArrayList<>();

        // iterate over the queue
        boolean isInterval = false; // isInterval: #Start seen > #End seen
        boolean isGap = false;      // isGap:      #GapStart seen > #GapEnd seen
        long intervalStart = 0;
        for (AnnotatedPoint point : queue) {
            switch (point.type) {
                case Start:
                    if (!isGap) {
                        intervalStart = point.value;
                    }
                    isInterval = true;
                    break;
                case End:
                    if (!isGap) {
                        result.add(new Interval(intervalStart, point.value));
                    }
                    isInterval = false;
                    break;
                case GapStart:
                    if (isInterval) {
                        result.add(new Interval(intervalStart, point.value));
                    }
                    isGap = true;
                    break;
                case GapEnd:
                    if (isInterval) {
                        intervalStart = point.value;
                    }
                    isGap = false;
                    break;
            }
        }

        return result;

    }
}
