package ru.yandex.qe.mail.meetings.cron;

import java.text.DateFormat;
import java.text.SimpleDateFormat;

import javax.mail.MessagingException;

import freemarker.template.Configuration;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailPreparationException;
import org.springframework.mail.javamail.MimeMessageHelper;

import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

/**
 * @author Sergey Galyamichev
 */
public abstract class AbstractMessageBuilder {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractMessageBuilder.class);
    public static final String AT_YANDEX = "@yandex-team.ru"; // should be substituted with staff call

    protected final static String NAME = "name";
    protected static final ThreadLocal<DateFormat> DF_HOLDER =
            ThreadLocal.withInitial(() -> new SimpleDateFormat("d.MM"));

    public static final boolean SEND_REAL_EMAILS = Boolean.getBoolean("send.real.email");
    private static final String DEFAULT_YANDEX_USER = System.getProperty("default.yandex.user", "selivanov@yandex-team.com");

    protected final Configuration configuration;
    protected final NotificationConfiguration notConfig;
    protected final StaffClient staffClient;

    public AbstractMessageBuilder(Configuration configuration, StaffClient staffClient, NotificationConfiguration notConfig) {
        this.configuration = configuration;
        this.notConfig = notConfig;
        this.staffClient = staffClient;
    }

    protected void prepareMessageHeadersDirect(MimeMessageHelper message, String email, String cc) throws MessagingException {
        message.setFrom(notConfig.getFrom());
        if (SEND_REAL_EMAILS) {
            message.setTo(email);
            if (StringUtils.isNotBlank(cc)) { // no cc in dev & qa
                message.setCc(cc);
            }
        } else {
            message.setTo(DEFAULT_YANDEX_USER);
        }
        message.setReplyTo(notConfig.getReplyTo());
    }

    protected void prepareMessageHeaders(MimeMessageHelper message, String email, String cc) throws MessagingException {
        message.setFrom(notConfig.getFrom());
        Person person = staffClient.getByLogin(getLogin(email));
        if (person == null) {
            throw new MailPreparationException("Skip non staff... for email: " + email + " .");
        }
        if (person.getAssitant() != null) {
            LOG.info("Assistant involved {}!!!", getAssistantEmail(person));
        }
        if (SEND_REAL_EMAILS) {
            if (person.getAssitant() != null) {
//                message.setTo(getAssistantEmail(person));
                throw new MailPreparationException("Mails to assistants will be ready soon...");
            } else {
                message.setTo(email);
            }
            if (StringUtils.isNotBlank(cc)) { // no cc in dev & qa
                message.setCc(cc);
            }
        } else {
            message.setTo(DEFAULT_YANDEX_USER);
        }
        message.setReplyTo(notConfig.getReplyTo());
    }

    private String getAssistantEmail(Person person) {
        return person.getAssitant().contains("@") ? person.getAssitant() : person.getAssitant() + AT_YANDEX;
    }

    public static String getLogin(String email) {
        return email.substring(0, email.indexOf('@'));
    }
}
