package ru.yandex.qe.mail.meetings.cron.absence;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.codahale.metrics.MetricRegistry;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.analisys.EventGapAnalyzer;
import ru.yandex.qe.mail.meetings.api.resource.CalendarActions;
import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.cron.dismissed.LoginWhitelist;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Events;
import ru.yandex.qe.mail.meetings.services.gaps.GapApi;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gap;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gaps;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * @author Sergey Galyamichev
 */
@Component
public class AbsenceCheckerJob implements Job {
    private static final Logger LOG = LoggerFactory.getLogger(AbsenceCheckerJob.class);
    @Inject
    private LoginWhitelist logins;
    @Inject
    private GapApi gapApi;
    @Inject
    private StaffClient staffClient;
    @Inject
    private CalendarWeb calendarWeb;
    @Inject
    private GapEventMessageBuilder messageBuilder;
    @Inject
    private SentEmailsDao sentEmailsDao;
    @Inject
    private JavaMailSender mailSender;
    @Inject
    private MetricRegistry metricRegistry;

    @Override
    public void execute(JobExecutionContext context) {
        LOG.info("AbsenceCheckerJob has been started...");
        Date fireTime = context.getFireTime();
        DateRange toNextWeek = DateRange.range(fireTime, DateRange.nextBusinessWeek(fireTime).getTo());
        GapActionsBuilder gapActionsBuilder = new GapActionsBuilder(fireTime);
        notifyEvents(toNextWeek, gapActionsBuilder);
        LOG.info("AbsenceCheckerJob has been done.");
    }

    public void notifyEvents(DateRange scanPeriod, GapActionsBuilder gapActionsBuilder) {
        Gaps gaps = gapApi.exportGaps(scanPeriod.getFrom(), scanPeriod.getTo(),null);
        Map<String, List<CalendarAction>> existedActions = sentEmailsDao.getActions(scanPeriod).stream()
                .collect(Collectors.groupingBy(CalendarAction::getEmail));
        for (Map.Entry<String, List<Gap>> entry : gaps.getPersons().entrySet()) {
            Set<CalendarAction> actions = new HashSet<>();
            List<Gap> nextWeekGaps = new ArrayList<>();
            String login = entry.getKey();
            if (!logins.isLucky(login)) {
                continue;
            }
            Person person = staffClient.getByLogin(login);
            if (person.getAssitant() != null) {
                continue;
            }
            for (Gap gap : entry.getValue()) {
                if (EventGapAnalyzer.isAbsence(gap) && scanPeriod.contains(gap.getDateFrom())) {
                    Events events = calendarWeb.getEvents(person.getUid(), gap.getDateFrom(), gap.getDateTo());
                    List<Event> eventList = events.getEvents().stream()
                            .filter(e -> EventGapAnalyzer.isIn(gap, e))
                            .filter(e -> !EventGapAnalyzer.isAGap(gap, e))
                            .filter(e -> !notified(existedActions.getOrDefault(login + AbstractMessageBuilder.AT_YANDEX, Collections.emptyList()), e))
                            .collect(Collectors.toList());
                    if (!eventList.isEmpty()) {
                        actions.addAll(gapActionsBuilder.buildActions(person, eventList));
                        nextWeekGaps.add(gap);
                    }
                }
            }
            if (!actions.isEmpty()) {
                try {
                    LOG.info("Going to send email to {} ", person.getLogin());
                    actions.forEach(a -> LOG.info("Action: {}", a));
                    Map<ActionType, List<CalendarAction>> actionByType = setGroupId(actions);
//todo:GREG-947                    mailSender.send(mimeMessage -> messageBuilder.prepareMessage(mimeMessage, person, nextWeekGaps, actionByType));
//todo:GREG-947                    report(actions);
//todo:GREG-947                    sentEmailsDao.insertActions(actions);
                } catch (MailException e) {
                    LOG.error("Email hasn't been sent", e);
                }
            }
        }
    }

    private void report(Collection<CalendarAction> actions) {
        if (metricRegistry != null) {
            actions.forEach(action -> metricRegistry.counter(MetricRegistry.name(CalendarActions.class, action.getType().name().toLowerCase(),
                    action.getStatus().name().toLowerCase())).inc());
        }
    }

    public Map<ActionType, List<CalendarAction>> setGroupId(Collection<CalendarAction> actions) {
        Map<ActionType, String> actionType2GroupIdMap = new HashMap<>();
        for (CalendarAction action : actions) {
            actionType2GroupIdMap.putIfAbsent(action.getType(), UUID.randomUUID().toString());
        }
        return actions.stream()
                .peek(a -> a.setGroupId(actionType2GroupIdMap.get(a.getType())))
                .collect(Collectors.groupingBy(CalendarAction::getType));
    }

    private boolean notified(List<CalendarAction> actions, Event e) {
        return actions.stream()
                .anyMatch(a -> Objects.equals(a.getEventId(), e.getEventId()) &&
                        Objects.equals(a.getInstanceStartTs(), e.getInstanceStartTs()) &&
                        GapActionsBuilder.isPossibleType(a.getType()));
    }
}
