package ru.yandex.qe.mail.meetings.cron.absence;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.cron.dismissed.CalendarActionNotification;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Decision;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventUser;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * @author Sergey Galyamichev
 */
public class GapActionsBuilder {
    private static final int TTL = 5;

    private final int ttl;
    private final Date date;

    public GapActionsBuilder(Date date) {
        ttl = TTL;
        this.date = date;
    }

    /**
     * Build actions for evey that user gonna miss
     *
     * @param person with gap
     * @param events user probably miss
     * @return suggested actions
     */
    public List<CalendarAction> buildActions(Person person, List<Event> events) {
        List<CalendarAction> actions = new ArrayList<>();
        for (Event event : events) {
            if (event.getResources().isEmpty() || CalendarHelper.isBigEvent(event)) {
                continue;
            }
            final EventUser organizer = event.getOrganizer();
            if (organizer != null && organizer.getLogin().equals(person.getLogin())) {
                List<CalendarAction> proposed = proposeActions(person, event);
                if (!proposed.isEmpty()) {
                    actions.addAll(proposed);
                } else {
                    actions.add(buildNotification(person, event));
                }
            } else {
                actions.add(buildDecisionNoAction(person, event));
            }
        }
        return actions;
    }

    private List<CalendarAction> proposeActions(Person person, Event event) {
        long realAttendees = event.getAttendees().stream()
                .map(EventUser::getDecision)
                .filter(d -> d == Decision.YES)
                .count();
        if (realAttendees <= 1) {
            return Collections.singletonList(buildRemoveEvent(person, event));
        } else {
            // todo more precise cases
            return Collections.emptyList();
        }
    }

    private CalendarAction buildRemoveEvent(Person person, Event event) {
        CalendarAction action = buildAction(date, person.getWorkEmail(), ttl, event);
        action.setType(ActionType.REMOVE_EVENT);
        return action;
    }

    private CalendarAction buildNotification(Person person, Event event) {
        CalendarAction action = buildAction(date, person.getWorkEmail(), ttl, event);
        action.setType(ActionType.NOTIFICATION);
        action.setStatus(Status.SENT);
        return action;
    }

    private CalendarAction buildDecisionNoAction(Person person, Event event) {
        CalendarAction action = buildAction(date, person.getWorkEmail(), ttl, event);
        action.setType(ActionType.DECLINE_EVENT);
        return action;
    }

    static boolean isPossibleType(ActionType type) {
        return type == ActionType.DECLINE_EVENT || type == ActionType.REMOVE_EVENT || type == ActionType.NOTIFICATION;
    }

    static CalendarAction buildAction(Date date, String email, int ttlDays, Event event) {
        CalendarActionNotification action = new CalendarActionNotification();
        action.setEvent(event);
        action.setEventId(event.getEventId());
        action.setEmail(email);
        action.setCreateDate(date);
        action.setTriggerTime(DateRange.addInterval(event.getStart(), Calendar.DAY_OF_MONTH, ttlDays));
        action.setStart(event.getStart());
        action.setName(event.getName());
        action.setSequence(event.getSequence());
        action.setInstanceStartTs(event.getInstanceStartTs());
        action.setStatus(Status.PENDING);
        action.setActionId(UUID.randomUUID().toString());
        return action;
    }
}
