package ru.yandex.qe.mail.meetings.cron.absence;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import freemarker.template.Configuration;
import org.apache.commons.lang3.CharEncoding;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.services.gaps.dto.Gap;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Language;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

/**
 * @author Sergey Galyamichev
 */
@Component
public class GapEventMessageBuilder extends AbstractMessageBuilder {
    private static final String TEMPLATE = "user_absence.ftl";
    private final String host;

    public GapEventMessageBuilder(@Value("${meetings.host.url}") String host, Configuration configuration, StaffClient staffClient, NotificationConfiguration notConfig) {
        super(configuration, staffClient, notConfig);
        this.host = host;
    }

    public void prepareMessage(MimeMessage mimeMessage, Person person, List<Gap> gaps, Map<ActionType, List<CalendarAction>> actionByType) throws MessagingException {
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, CharEncoding.UTF_8);
        prepareMessageHeaders(message, person.getWorkEmail(), notConfig.getDismissed().getCc());
        message.setSubject(getLogin(person.getWorkEmail()) + "@ " + notConfig.getGapsSubject());
        message.setText(buildBody(person, gaps, actionByType), true);
    }

    private String buildBody(Person person, List<Gap> gaps, Map<ActionType, List<CalendarAction>> actions) throws MessagingException {
        try {
            Map<String, Object> valuesMap = new HashMap<>();
            valuesMap.put(NAME, person.getName().toString(Language.RUSSIAN));
            valuesMap.put("gaps", gaps);
            valuesMap.put("host", host);
            valuesMap.put("notificationsGroupId", findGroupId(actions, ActionType.NOTIFICATION));
            valuesMap.put("notifications", actions.get(ActionType.NOTIFICATION));
            valuesMap.put("removesGroupId", findGroupId(actions, ActionType.REMOVE_EVENT));
            valuesMap.put("removes", actions.get(ActionType.REMOVE_EVENT));
            valuesMap.put("declinesGroupId", findGroupId(actions, ActionType.DECLINE_EVENT));
            valuesMap.put("declines", actions.get(ActionType.DECLINE_EVENT));
            return FreeMarkerTemplateUtils.processTemplateIntoString(configuration.getTemplate(TEMPLATE), valuesMap);
        } catch (Exception e) {
            throw new MessagingException("Unable to build body", e);
        }
    }

    private String findGroupId(Map<ActionType, List<CalendarAction>> actions, ActionType actionType) {
        return actions.values().stream()
                .flatMap(List::stream)
                .filter(calendarAction -> calendarAction.getType() == actionType)
                .findFirst()
                .map(CalendarAction::getGroupId)
                .orElse("-");
    }
}
