package ru.yandex.qe.mail.meetings.cron.actions;

import java.util.List;
import java.util.stream.Collectors;

import javax.ws.rs.NotFoundException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Decision;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.ReplyData;
import ru.yandex.qe.mail.meetings.services.calendar.dto.WebEventData;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;

/**
 * @author Sergey Galyamichev
 */
@Component
public class ActionsRunner {
    private static final Logger LOG = LoggerFactory.getLogger(ActionsRunner.class);
    private final CalendarUpdate calendarUpdate;
    private final CalendarWeb calendarWeb;
    private final CalendarHelper calendar;
    private final StaffClient staffClient;

    public ActionsRunner(CalendarUpdate calendarUpdate, CalendarWeb calendarWeb, CalendarHelper calendar, StaffClient staffClient) {
        this.calendarUpdate = calendarUpdate;
        this.calendarWeb = calendarWeb;
        this.calendar = calendar;
        this.staffClient = staffClient;
    }

    public Status perform(CalendarAction action) {
        LOG.info("performing ({})", action.toString());
        if (action.getStatus() != Status.ACCEPTED && action.getStatus() != Status.PENDING) {
            return action.getStatus();
        }
        ActionType type = action.getType();
        Status status = Status.PERFORMED;
        Event event;
        switch (type) {
            case REMOVE_EVENT:
                calendar.doCall(() -> calendarUpdate.deleteEvent(action.getEventId(), action.getSequence(), action.getInstanceStartTs(), false, null));
                break;
            case REMOVE_SERIES:
                event = calendar.getBaseEvent(action.getEventId());
                calendar.doCall(() -> calendarUpdate.deleteEvent(event.getEventId(), action.getSequence(), null, true, null));
                break;
            case REMOVE_PARTICIPANTS:
                event = calendarWeb.getEvent(action.getEventId());
                status = removeParticipants(event, action, event.getInstanceStartTs(), false);
                break;
            case REMOVE_PARTICIPANTS_SERIES:
                event = calendar.getSeriesInstance(action.getEventId());
                status = removeParticipants(event, action, event.getInstanceStartTs(), true);
                break;
            case DECLINE_EVENT:
                ReplyData data = new ReplyData(action.getEventId(), Decision.NO, action.getInstanceStartTs(), false, action.getContext());
                String uid = staffClient.getByLogin(AbstractMessageBuilder.getLogin(action.getEmail())).getUid();
                calendarUpdate.handleReply(uid, data);
                break;
            case SWAP_RESOURCE:
            case MOVE_RESOURCE:
                status = new ResourceSwapper(calendarWeb, staffClient).execute(action);
                break;
            case NOTIFICATION:
                status = action.getStatus();
                break;
        }
        return status;
    }

    private Status removeParticipants(Event event, CalendarAction action, String instanceStartTs, boolean applyToFuture) {
        try {
            int original = event.getAttendees().size();
            List<String> logins = checkReturnee(action.getLogins());
            event.getAttendees().removeIf(user -> logins.contains(user.getLogin()));
            if (original != event.getAttendees().size()) {
                WebEventData data = WebEventData.fromEvent(event);
                calendar.doCall(() -> calendarUpdate.updateEvent(event.getEventId(), event.getSequence(), instanceStartTs, applyToFuture, false, null, data));
                return Status.PERFORMED;
            }
            return Status.CHANGED;
        } catch (NotFoundException e) {
            LOG.warn("Event removed: {}", e.getMessage());
            return Status.FAILED;
        }
    }

    private List<String> checkReturnee(List<String> candidates) {
        return candidates.stream()
                .map(staffClient::getByLogin)
                .filter(p -> p.getOfficial().isDismissed())
                .map(Person::getLogin)
                .collect(Collectors.toList());
    }
}
