package ru.yandex.qe.mail.meetings.cron.actions;

import java.util.Arrays;
import java.util.Collections;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;

/**
 * @author Sergey Galyamichev
 */
public class Contexts {
    public static class Swap {
        private static final int REQUESTER_INDEX = 0;
        private static final int TARGET_EVENT_INDEX = 1;
        private static final int RESOURCE_1_INDEX = 2;
        private static final int RESOURCE_2_INDEX = 3;

        private final String organizer;
        private final int targetEventId;
        private final String sourceResource;
        private final String targetResource;

        private Swap(String organizer, int targetEventId, String sourceResource, String targetResource) {
            this.organizer = organizer;
            this.targetEventId = targetEventId;
            this.sourceResource = sourceResource;
            this.targetResource = targetResource;
        }

        public String getOrganizer() {
            return organizer;
        }

        public int getTargetEventId() {
            return targetEventId;
        }

        public String getSourceResource() {
            return sourceResource;
        }

        public String getTargetResource() {
            return targetResource;
        }

        public static Swap parse(String context) {
            String[] params = context.split(CalendarAction.CONTEXT_DELIMITER);
            String organizer = params[REQUESTER_INDEX];
            int targetEventId = Integer.parseInt(params[TARGET_EVENT_INDEX]);
            String sourceResource = params[RESOURCE_1_INDEX].isBlank() ? null : params[RESOURCE_1_INDEX];
            String targetResource = params[RESOURCE_2_INDEX];
            return new Swap(organizer, targetEventId, sourceResource, targetResource);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Swap swap = (Swap) o;
            return targetEventId == swap.targetEventId &&
                    Objects.equals(organizer, swap.organizer) &&
                    Objects.equals(sourceResource, swap.sourceResource) &&
                    Objects.equals(targetResource, swap.targetResource);
        }

        @Override
        public int hashCode() {
            return Objects.hash(organizer, targetEventId, sourceResource, targetResource);
        }

        @Override
        public String toString() {
            return  getOrganizer() + CalendarAction.CONTEXT_DELIMITER +
                    targetEventId + CalendarAction.CONTEXT_DELIMITER +
                    (sourceResource == null ? "" : sourceResource) + CalendarAction.CONTEXT_DELIMITER +
                    targetResource;
        }
    }

    public static class Scan {
        private final static String AUTO = "auto";
        private final static String CHANGE = "change";
        private final static String OFFICE_DELIMITER = ",";
        private final static int OFFICE_ID_INDEX = 0;
        private final static int FILTER_INDEX = 1;
        private final static int CHANGE_INDEX = 2;

        private final boolean isAuto;
        private final Set<Integer> officeId;
        private final boolean change;
        private final String filter;

        public Scan(boolean isAuto, boolean isChange, Set<Integer> officeId, String filter) {
            this.isAuto = isAuto;
            this.officeId = officeId;
            this.change = isChange;
            this.filter = filter;
        }

        public boolean isAuto() {
            return isAuto;
        }

        public boolean isChange() {
            return change;
        }

        public Set<Integer> getOfficeIds() {
            return officeId;
        }

        public String getOffices() {
            return officeId.stream()
                    .map(Objects::toString)
                    .collect(Collectors.joining(OFFICE_DELIMITER));
        }

        public String getFilter() {
            return filter;
        }

        public static Scan parse(String context) {
            String[] params = context.split(CalendarAction.CONTEXT_DELIMITER, 2);
            String filter = params.length > 1 && !params[FILTER_INDEX].isBlank() ? params[FILTER_INDEX] : null;
            boolean isChange = params.length > 2 && CHANGE.equals(params[CHANGE_INDEX]);
            if (AUTO.equals(params[OFFICE_ID_INDEX])) {
                return new Scan(true, isChange, Collections.emptySet(), filter);
            } else {
                Set<Integer> officeIds = parseOffices(params[OFFICE_ID_INDEX]);
                return new Scan(false, isChange, officeIds, filter);
            }
        }

        public static Set<Integer> parseOffices(String param) {
            return Arrays.stream(param.split(","))
                                .map(Integer::parseInt)
                                .collect(Collectors.toSet());
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Scan scan = (Scan) o;
            return isAuto == scan.isAuto &&
                    Objects.equals(officeId, scan.officeId) &&
                    Objects.equals(filter, scan.filter);
        }

        @Override
        public int hashCode() {
            return Objects.hash(isAuto, officeId, filter);
        }

        @Override
        public String toString() {
            String office = isAuto ? AUTO : getOffices();
            String result;
            if (change) {
                result = office + CalendarAction.CONTEXT_DELIMITER + (filter == null ? "" : filter ) +
                        CalendarAction.CONTEXT_DELIMITER + CHANGE;
            } else {
                result = filter != null ? office + CalendarAction.CONTEXT_DELIMITER + filter: office;
            }
            return result;
        }
    }

    public static Swap swap(String organizer, int targetEventId, String sourceResource, String targetResource) {
        return new Swap(organizer, targetEventId, sourceResource, targetResource);
    }

    public static Scan scan(boolean isAuto, boolean isChange, Set<Integer> officeId, String filter) {
        return new Scan(isAuto, isChange, officeId, filter);
    }
}
