package ru.yandex.qe.mail.meetings.cron.actions;

import java.io.IOException;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.qe.mail.meetings.api.resource.dto.MergeEventRequest;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.calendar.dto.WebEventData;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.ws.EventResourceDescriptor;
import ru.yandex.qe.mail.meetings.ws.validation.ValidationResult;

/**
 * @author Sergey Galyamichev
 */
public class MergeResource {
    private CalendarWeb calendarWeb;
    private StaffClient staffClient;
    private String organizer;

    private Event mainEvent;
    private Event otherEvent;
    private Person requester;

    public MergeResource(CalendarWeb calendarWeb, StaffClient staffClient, String organizer) {
        this.calendarWeb = calendarWeb;
        this.staffClient = staffClient;
        this.organizer = organizer;
    }

    public Event parseUrl(String url, String date) throws IOException {
        EventResourceDescriptor main = EventResourceDescriptor.fromRequest(calendarWeb, staffClient, organizer, url);
        return main.findEvent(calendarWeb, staffClient, date);
    }

    public ValidationResult validate(MergeEventRequest request) {
        ValidationResult result = ValidationResult.success();
        try {
            mainEvent = parseUrl(request.getMainEventUrl(), request.getEventDate());
        } catch (CalendarException ce) {
            result = ValidationResult.merge(result, ValidationResult.error("mainEventUrl", ce));
        } catch (Exception e) {
            result = ValidationResult.merge(result, ValidationResult.error("mainEventUrl", ValidationResult.INVALID_URL));
        }
        try {
            otherEvent = parseUrl(request.getOtherEventUrl(), request.getEventDate());
        } catch (CalendarException ce) {
            result = ValidationResult.merge(result, ValidationResult.error("otherEventUrl", ce));
        } catch (Exception e) {
            result = ValidationResult.merge(result, ValidationResult.error("otherEventUrl", ValidationResult.INVALID_URL));
        }
        if (result.getStatus() == ValidationResult.Status.ERROR) {
            return result;
        }
        if (organizer != null) {
            requester = staffClient.getByLogin(organizer);
            if (!checkMainPermission(requester, mainEvent.getEventId())) {
                result = ValidationResult.merge(result, ValidationResult.error("mainEventUrl", "No permissions to modify event"));
            }
            if (!checkOtherPermission(requester, otherEvent.getEventId())) {
                result = ValidationResult.merge(result, ValidationResult.error("otherEventUrl", "No permissions to modify event"));
            }
        }
        if (!mainEvent.getEnd().equals(otherEvent.getStart()) && !mainEvent.getStart().equals(otherEvent.getEnd())) {
            ValidationResult.merge(result, ValidationResult.error("mainEventUrl", "Event aren't connected"));
        }
        if (!resourceEmails(mainEvent).equals(resourceEmails(otherEvent))) {
            result = ValidationResult.merge(result, ValidationResult.error("mainEventUrl", "Resources are different"));
        }
        return result;
    }

    public boolean checkMainPermission(Person person, int eventId) {
        Event event = calendarWeb.getEvent(eventId, person.getUid());
        return event.getActions().isEdit() && event.getActions().isMove();
    }

    public boolean checkOtherPermission(Person person, int eventId) {
        Event event = calendarWeb.getEvent(eventId, person.getUid());
        return event.getActions().isDelete();
    }

    public void execute(CalendarUpdate calendarUpdate, MergeEventRequest request) {
        if (validate(request).getStatus() == ValidationResult.Status.OK && requester != null) {
            WebEventData mainEventWeb = WebEventData.fromEvent(mainEvent);
            merge(mainEventWeb, otherEvent);
            calendarUpdate.deleteEvent(otherEvent.getEventId(), otherEvent.getSequence(), otherEvent.getInstanceStartTs(), false, requester.getUid()
            );
            calendarUpdate.updateEvent(mainEvent.getEventId(), mainEvent.getSequence(), mainEvent.getInstanceStartTs(), false, true, requester.getUid(),
                    mainEventWeb);
        }
    }

    private static void merge(WebEventData mainEventWeb, Event otherEvent) {
        if (mainEventWeb.getEndTs().equals(otherEvent.getStart())) {
            mainEventWeb.setEndTs(otherEvent.getEnd());
        } else if (mainEventWeb.getStartTs().equals(otherEvent.getEnd())) {
            mainEventWeb.setStartTs(otherEvent.getStart());
        }
    }

    private static Set<String> resourceEmails(Event otherEvent) {
        return otherEvent.getResources().stream()
                .map(Resource.Info::getEmail)
                .collect(Collectors.toSet());
    }
}
