package ru.yandex.qe.mail.meetings.cron.actions;

import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.api.resource.dto.SwapResourceRequest;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.DateRange;
import ru.yandex.qe.mail.meetings.utils.FormConverters;
import ru.yandex.qe.mail.meetings.ws.EventResourceDescriptor;
import ru.yandex.qe.mail.meetings.ws.validation.ValidationResult;

/**
 * @author Sergey Galyamichev
 */
public class ResourceSwapper {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceSwapper.class);

    private CalendarWeb calendarWeb;
    private StaffClient staffClient;

    private String organizer;
    private Event sourceEvent;
    private Event targetEvent;
    private Resource.Info sourceResource;
    private Resource.Info targetResource;
    private boolean move;

    public Event getSourceEvent() {
        return sourceEvent;
    }

    public Event getTargetEvent() {
        return targetEvent;
    }

    public Resource.Info getSourceResource() {
        return sourceResource;
    }

    public Resource.Info getTargetResource() {
        return targetResource;
    }

    public ResourceSwapper(CalendarWeb calendarWeb, StaffClient staffClient) {
        this.staffClient = staffClient;
        this.calendarWeb = calendarWeb;
    }

    public CalendarAction prepare(SwapResourceRequest request, String requesterLogin) {
        this.organizer = requesterLogin;
        ValidationResult result = validate(request);
        if (result.getStatus() == ValidationResult.Status.OK) {
            boolean allowed = move ? checkInvitePermission(organizer, sourceEvent.getEventId()) :
                    checkEditPermission(organizer, sourceEvent.getEventId());
            if (allowed) {
                return buildAction(organizer, sourceResource, targetResource);
            } else {
                throw new IllegalArgumentException("No permission to edit event");
            }
        } else {
            LOG.error("Validation result: {}", result);
            throw new IllegalArgumentException("Request isn't valid!");
        }
    }

    public boolean checkInvitePermission(String login, int eventId) {
        Person person = staffClient.getByLogin(login);
        Event event = calendarWeb.getEvent(eventId, person.getUid());
        return event.getActions().isInvite();
    }

    public boolean checkEditPermission(String login, int eventId) {
        Person person = staffClient.getByLogin(login);
        Event event = calendarWeb.getEvent(eventId, person.getUid());
        return event.getActions().isEdit();
    }

    public ValidationResult validate(SwapResourceRequest request) {
        ValidationResult result = ValidationResult.success();
        move = FormConverters.convertYesNo(request.getMoveResource());
        try {
            sourceEvent = parseUrl(request.getSourceCalendarUrl(), request.getEventDate());
        } catch (CalendarException ce) {
            result = ValidationResult.merge(result, ValidationResult.error("sourceCalendarUrl", ce));
        } catch (Exception e) {
            result = ValidationResult.merge(result, ValidationResult.error("sourceCalendarUrl", ValidationResult.INVALID_URL));
        }
        try {
            targetEvent = parseUrl(request.getTargetCalendarUrl(), request.getEventDate());
        } catch (CalendarException ce) {
            result = ValidationResult.merge(result, ValidationResult.error("targetCalendarUrl", ce));
        } catch (Exception e) {
            result = ValidationResult.merge(result, ValidationResult.error("targetCalendarUrl", ValidationResult.INVALID_URL));
        }
        Integer officeId = FormConverters.convertOne(request.getOffices(), FormConverters.OFFICES);
        if (officeId == null) {
            result = ValidationResult.merge(result, ValidationResult.error("offices", "Selected unknown office"));
        }
        if (officeId != null && result.getStatus() == ValidationResult.Status.OK) {
            List<Resource.Info> sourceInfos = find(sourceEvent.getResources(), officeId);
            List<Resource.Info> targetInfos = find(targetEvent.getResources(), officeId);
            if (sourceInfos.size() == 0 && targetInfos.size() == 0) {
                result = ValidationResult.merge(result, ValidationResult.error("offices", "No resources in office"));
            } else {
                if (targetInfos.size() > 1) {
                    result = ValidationResult.merge(result, ValidationResult.error("targetCalendarUrl", "More than one resource in the office"));
                }
                if (sourceInfos.size() > 1) {
                    result = ValidationResult.merge(result, ValidationResult.error("sourceCalendarUrl", "More than one resource in the office"));
                }
                if (sourceInfos.size() == 0) {
                    if (!move) {
                        result = ValidationResult.merge(result, ValidationResult.error("sourceCalendarUrl", "No resource in office"));
                    }
                } else {
                    sourceResource = sourceInfos.get(0);
                }
                if (targetInfos.size() == 0) {
                    result = ValidationResult.merge(result, ValidationResult.error("targetCalendarUrl", "No resource in office"));
                } else {
                    targetResource = targetInfos.get(0);
                }
            }
            if (sourceEvent.getEventId().intValue() == targetEvent.getEventId().intValue()) {
                result = ValidationResult.error("targetCalendarUrl", "Same events");
            }
            if (!sourceEvent.getStart().equals(targetEvent.getStart()) || !sourceEvent.getEnd().equals(targetEvent.getEnd())) {
                result = ValidationResult.error("targetCalendarUrl", "Events timeframes are different");
            }
        }
        return result;
    }

    public CalendarAction buildAction(String organizer, Resource.Info sourceResource, Resource.Info targetResource) {
        CalendarAction action = new CalendarAction();
        action.setEmail(targetEvent.getOrganizer().getLogin() + AbstractMessageBuilder.AT_YANDEX);
        action.setCreateDate(new Date());
        action.setStart(targetEvent.getStart());
        action.setName(targetEvent.getName());
        action.setSequence(targetEvent.getSequence());
        action.setInstanceStartTs(targetEvent.getInstanceStartTs());
        action.setTriggerTime(DateRange.tomorrow(targetEvent.getStart()));
        action.setType(move ? ActionType.MOVE_RESOURCE : ActionType.SWAP_RESOURCE);
        action.setEventId(targetEvent.getEventId());
        action.setActionId(UUID.randomUUID().toString());
        action.setContext(Contexts.swap(organizer, sourceEvent.getEventId(), sourceResource == null ? null : sourceResource.getEmail(), targetResource.getEmail()).toString());
        return action;
    }

    public Event parseUrl(String url, String date) throws IOException {
        EventResourceDescriptor desc = EventResourceDescriptor.fromRequest(calendarWeb, staffClient, organizer, url);
        return desc.findEvent(calendarWeb, staffClient, date);
    }

    private List<Resource.Info> find(List<Resource.Info> resources, int officeId) {
        return resources.stream()
                .filter(r -> r.getOfficeId() == officeId)
                .collect(Collectors.toList());
    }

    public Status execute(CalendarAction action) {
        try {
            Contexts.Swap swap = Contexts.Swap.parse(action.getContext());
            Person requester = staffClient.getByLogin(swap.getOrganizer());
            Person accepter = staffClient.getByLogin(AbstractMessageBuilder.getLogin(action.getEmail()));
            calendarWeb.moveResource(accepter.getUid(), requester.getUid(), action.getEventId(), swap.getTargetEventId(), swap.getTargetResource(), action.getInstanceStartTs());
            if (action.getType() == ActionType.SWAP_RESOURCE) {
                calendarWeb.moveResource(requester.getUid(), accepter.getUid(), swap.getTargetEventId(), action.getEventId(), swap.getSourceResource(), action.getInstanceStartTs());
            }
            return Status.PERFORMED;
        } catch (Exception e) {
            LOG.warn("Event removed: {}", e.getMessage());
            return Status.FAILED;
        }

    }

    public Person getRequester() {
        return staffClient.getByLogin(organizer);
    }

    public Status tryExecute(CalendarAction action) {
        if (checkEditPermission(organizer, targetEvent.getEventId())) {
            return execute(action);
        }
        return Status.PENDING;
    }

}
