package ru.yandex.qe.mail.meetings.cron.actions.email;

import java.util.HashMap;
import java.util.Map;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import freemarker.template.Configuration;
import org.apache.commons.lang3.CharEncoding;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.cron.actions.ResourceSwapper;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;

/**
 * @author Sergey Galyamichev
 */
@Component
public class SwapRequestMessageBuilder extends AbstractMessageBuilder {
    private static final String SWAP_TEMPLATE = "swap_resource.ftl";
    private static final String MOVE_TEMPLATE = "move_resource.ftl";
    private final String host;

    public SwapRequestMessageBuilder(@Value("${meetings.host.url}") String host,
                                Configuration configuration,
                                StaffClient staffClient,
                                NotificationConfiguration notConfig) {
        super(configuration, staffClient, notConfig);
        this.host = host;
    }

    public void prepareMessage(MimeMessage mimeMessage, CalendarAction action, ResourceSwapper swapper, String description) throws MessagingException {
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, CharEncoding.UTF_8);
        Person person = staffClient.getByLogin(getLogin(action.getEmail()));
        prepareMessageHeaders(message, action.getEmail(), null /*organizer + AT_YANDEX*/);
        message.setSubject(notConfig.getResourceSwap());
        message.setText(buildBody(person, action, swapper, description), true);
    }

    private String buildBody(Person person, CalendarAction action, ResourceSwapper swapper, String description) throws MessagingException {
        try {
            Map<String, Object> valuesMap = new HashMap<>();
            valuesMap.put(NAME, person.getName().toString(person.getLanguage().getUi()));
            valuesMap.put("action", action);
            valuesMap.put("host", host);
            valuesMap.put("who", swapper.getRequester().getName().toString(person.getLanguage().getUi()));
            valuesMap.put("targetEvent", swapper.getTargetEvent());
            valuesMap.put("targetResource", swapper.getTargetResource());
            valuesMap.put("targetResourceId", getLogin(swapper.getTargetResource().getEmail()));
            if (action.getType() == ActionType.SWAP_RESOURCE) {
                valuesMap.put("sourceResource", swapper.getSourceResource());
                valuesMap.put("sourceResourceId", getLogin(swapper.getSourceResource().getEmail()));
            }
            valuesMap.put("login", getLogin(action.getEmail()));
            valuesMap.put("description", description);
            String template = action.getType() == ActionType.SWAP_RESOURCE ? SWAP_TEMPLATE : MOVE_TEMPLATE;
            return FreeMarkerTemplateUtils.processTemplateIntoString(configuration.getTemplate(template), valuesMap);
        } catch (Exception e) {
            throw new MessagingException("Unable to build body", e);
        }
    }
}
