package ru.yandex.qe.mail.meetings.cron.declines;

import java.util.List;
import java.util.Set;

import javax.inject.Inject;

import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.blamer.DeclineEvent;
import ru.yandex.qe.mail.meetings.blamer.DeclineEvents;
import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * The job sends email once a week to users that have more than <code>notifications.threshold</code> events declined
 * by display.
 *
 * @author Sergey Galiamichev
 */
@Component
public class DeclinedEventsJob implements Job {
    private static final Logger LOG = LoggerFactory.getLogger(DeclinedEventsJob.class);
    @Inject
    private NotificationConfiguration notConfig; // threshold should be moved somewhere
    @Inject
    private JavaMailSender mailSender;
    @Inject
    private DeclinedEventMessageBuilder messageBuilder;
    @Inject
    private DeclinedEventProvider eventProvider;
    @Inject
    private SentEmailsDao sentEmailsDao;

    @Override
    public void execute(JobExecutionContext context) {
        LOG.info("DeclinedEventsJob has been started...");
        DateRange lastWeek = DateRange.lastBusinessWeek(context.getFireTime());
        List<DeclineEvents> eventList = eventProvider.getDeclineEvent(lastWeek, notConfig.getDeclined().getThreshold());
        Set<String> sent = sentEmailsDao.getSentEmails(lastWeek); // shouldn't be in events dao
        for (DeclineEvents event : eventList) {
            if (!sent.contains(event.getEmail())) {
                try {
                    if (sentEmailsDao.updateSent(event, lastWeek.getFrom())) {
                        List<DeclineEvent> events = eventProvider.getEventsList(event, lastWeek);
                        mailSender.send(mm -> messageBuilder.prepareMessage(mm, event, lastWeek, events));
                        LOG.debug("Email sent to " + event.getName());
                    } else {
                        LOG.error("Notification info wasn't saved!");
                    }
                } catch (MailException e) {
                    LOG.error("Email hasn't been sent", e);
                }
            }
        }
        LOG.info("DeclinedEventsJob has been done.");
    }
}
