package ru.yandex.qe.mail.meetings.cron.dismissed;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.analisys.UserCheck;
import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.User;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;
import ru.yandex.qe.mail.meetings.utils.DateRange;
import ru.yandex.qe.mail.meetings.ws.CalendarFacade;

/**
 * @author Sergey Galyamichev
 */
@Component
public class DismissedAttendeesEventsProvider {
    private final StaffClient staffClient;
    private final CalendarFacade calendarExporter;
    private final NotificationConfiguration notConfig;
    private final LoginWhitelist logins;

    public DismissedAttendeesEventsProvider(LoginWhitelist logins,
                                            StaffClient staffClient,
                                            CalendarFacade calendarExporter,
                                            NotificationConfiguration notConfig) {
        this.logins = logins;
        this.staffClient = staffClient;
        this.calendarExporter = calendarExporter;
        this.notConfig = notConfig;
    }

    private User getUser(Pair<Event, List<User>> pair) {
        User organizer = pair.getLeft().getOrganizer();
        return new User(organizer.getName(), organizer.getEmail(), organizer.getLogin(), organizer.getOfficeId());
    }

    public Map<User, List<CalendarAction>> buildActions(@Nonnull Date fireTime, @Nonnull Date from, @Nonnull Date to) {
        Map<User, List<CalendarAction>> result = new HashMap<>();
        Set<Map.Entry<User, Map<Event, List<User>>>> entries = findDismissed(from, to).entrySet();
        for (Map.Entry<User, Map<Event, List<User>>> user : entries) {
            if (logins.isLucky(user.getKey().getLogin())) {
                List<CalendarAction> actions = buildActions(fireTime, user.getKey(), notConfig.getDismissed().getDaysToAction(), user.getValue());
                if (!actions.isEmpty()) {
                    result.put(user.getKey(), actions);
                }
            }
        }
        return result;
    }

    public Map<User, Map<Event, List<User>>> findDismissed(@Nonnull Date from, @Nonnull Date to) {
        UserCheck userCheck = new UserCheck.Bulk(staffClient);
        List<Pair<Event, List<User>>> events = calendarExporter.findDismissedAttendees(from, to, userCheck);
        return events.stream()
                .collect(Collectors.groupingBy(this::getUser,
                        Collectors.toMap(Pair::getLeft, Pair::getRight, (elder, younger) -> elder)));
    }

    static List<CalendarAction> buildActions(Date date, User organizer, int ttlDays, Map<Event, List<User>> events) {
        List<CalendarAction> actions = new ArrayList<>();
        String groupId = UUID.randomUUID().toString();
        for (Map.Entry<Event, List<User>> entry : events.entrySet()) {
            final Event event = entry.getKey();
            int totalAttendees = event.getAttendees().size() + event.getSubscribers().size();
            if (CalendarHelper.isVeryBigEvent(event)) {
                continue;
            }
            CalendarActionNotification action = new CalendarActionNotification();
            action.setEvent(event);
            action.setEventId(event.getEventId());
            if (totalAttendees - entry.getValue().size() > 0) {
                action.setType(ActionType.REMOVE_PARTICIPANTS_SERIES);
            } else {
                action.setType(ActionType.REMOVE_SERIES);
            }
            action.setEmail(organizer.getEmail());
            List<String> logins = entry.getValue().stream()
                    .map(User::getLogin)
                    .collect(Collectors.toList());
            action.setLogins(logins);
            action.setUsers(entry.getValue());
            action.setCreateDate(date);
            action.setTriggerTime(DateRange.addInterval(date, Calendar.DAY_OF_MONTH, ttlDays));
            action.setStart(event.getStart());
            action.setName(event.getName());
            action.setSequence(event.getSequence());
            action.setStatus(Status.PENDING);
            action.setActionId(UUID.randomUUID().toString());
            action.setGroupId(groupId);
            actions.add(action);
        }
        return actions;
    }
}
