package ru.yandex.qe.mail.meetings.cron.dismissed;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.services.calendar.dto.User;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * @author Sergey Galyamichev
 */
@Component
public class DismissedAttendeesJob implements Job {
    private static final Logger LOG = LoggerFactory.getLogger(DismissedAttendeesJob.class);
    @Inject
    private JavaMailSender mailSender;
    @Inject
    private DismissedAttendeesEventsProvider eventsProvider;
    @Inject
    private DismissedAttendeeMessageBuilder messageBuilder;
    @Inject
    private SentEmailsDao sentEmailsDao;

    @Override
    public void execute(JobExecutionContext context) {
        LOG.info("DismissedAttendeesJob has been started...");
        Date fireTime = context.getFireTime();
        DateRange nextWeek = DateRange.nextBusinessWeek(fireTime);
        Map<User, List<CalendarAction>> events = eventsProvider.buildActions(fireTime, nextWeek.getFrom(), nextWeek.getTo());
        Set<String> organizers = new HashSet<>();
        Set<String> dismissed = new HashSet<>();
        Map<String, Set<Integer>> notified = sentEmailsDao.getActions(Status.PENDING).stream()
                .collect(Collectors.groupingBy(CalendarAction::getEmail,
                        Collectors.mapping(CalendarAction::getEventId, Collectors.toSet())));
        for (Map.Entry<User, List<CalendarAction>> user : events.entrySet()) {
            organizers.add(user.getKey().getLogin());
            List<CalendarAction> actions = filterOutAlreadyNotified(user.getKey().getEmail(), notified, user.getValue());
            actions.forEach(a -> dismissed.addAll(a.getLogins()));
            if (!actions.isEmpty()) {
                try {
                    LOG.info("Going to send email to {} ", user.getKey().getLogin());
                    actions.forEach(a -> LOG.info("Action: {}", a));
                    mailSender.send(mimeMessage -> messageBuilder.prepareMessage(mimeMessage, nextWeek, user.getKey(), actions));
                    sentEmailsDao.insertActions(actions);
                } catch (MailException e) {
                    LOG.error("Email hasn't been sent", e);
                }
            }
        }
        LOG.info("Organizers size = {}, value = {}", organizers.size(), organizers.toString());
        LOG.info("Unprocessed dismissed users size = {}, value = {}", dismissed.size(), dismissed.toString());
        LOG.info("DismissedAttendeesJob has been done.");
    }

    private List<CalendarAction> filterOutAlreadyNotified(String email,
                                                          Map<String, Set<Integer>> notified,
                                                          List<CalendarAction> actions) {
        Set<Integer> events = notified.get(email);
        if (events == null) {
            return actions;
        } else {
            return Collections.emptyList();
//                actions.stream() // more correct
//                    .filter(a -> !events.contains(a.getEventId()))
//                    .collect(Collectors.toList());
        }
    }
}
