package ru.yandex.qe.mail.meetings.cron.scanner;

import java.util.Date;
import java.util.List;

import javax.inject.Inject;

import com.codahale.metrics.MetricRegistry;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.api.resource.CalendarActions;
import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * @author Sergey Galyamichev
 */
@Component
public class ResourceScannerJob implements Job {
    private static final Logger LOG = LoggerFactory.getLogger(ResourceScannerJob.class);
    @Inject
    private JavaMailSender mailSender;
    @Inject
    private SentEmailsDao sentEmailsDao;
    @Inject
    private ScanTimeoutMessageBuilder staleMessageBuilder;
    @Inject
    private FailureMessageBuilder failureMessageBuilder;
    @Inject
    private EventChangedMessageBuilder changedMessageBuilder;
    @Inject
    private SearchResourceTask searchResourceTask;
    @Inject
    private MetricRegistry metricRegistry;

    @Override
    public void execute(JobExecutionContext context) {
        LOG.info("ResourceScannerJob has been started...");
        List<CalendarAction> actions = sentEmailsDao.getActions(ActionType.ADD_RESOURCE, Status.ACCEPTED);
        LOG.info("Found {} pending actions...", actions.size());
        Date fireTime = context.getFireTime();
        for (CalendarAction action : actions) {
            try {
                Status status = executeAction(action, fireTime);
                if (status != action.getStatus()) {
                    if (sentEmailsDao.updateAction(action.getActionId(), status, action.getSequence())) {
                        report(status);
                    }
                }
            } catch (Exception e) {
                LOG.warn("Action skipped",e);
            }
        }
        LOG.info("ResourceScannerJob has been done.");
    }

    public Status executeAction(CalendarAction action, Date fireTime) {
        try {
            Status status = searchResourceTask.executeFor(action, fireTime);
            if (status != Status.ACCEPTED) {
                if (status == Status.CHANGED) {
                    LOG.info("Event for action {} was changed by user", action.toString());
                    mailSender.send(mimeMessage -> failureMessageBuilder.prepareMessage(mimeMessage, action, "All resource found"));
                }
                report(status);
            } else {
                if (action.getTriggerTime().compareTo(DateRange.toMSK(fireTime)) < 0) {
                    LOG.info("Action {} is outdated now", action.toString());
                    mailSender.send(mimeMessage -> staleMessageBuilder.prepareMessage(mimeMessage, action));
                    return Status.OUTDATED;
                }
            }
            return status;
        } catch (CalendarException e) {
            LOG.error("Unable update event", e);
        } catch (MailException e) {
            LOG.error("Email hasn't been sent", e);
        }
        mailSender.send(mimeMessage -> failureMessageBuilder.prepareMessage(mimeMessage, action, "Execution error"));
        return Status.FAILED;
    }

    public void report(Status status) {
        String name = MetricRegistry.name(CalendarActions.class, ActionType.ADD_RESOURCE.toString().toLowerCase(), status.name().toLowerCase());
        if (metricRegistry != null) {
            metricRegistry.counter(name).inc();
        }
    }
}
