package ru.yandex.qe.mail.meetings.cron.scanner;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.cron.actions.Contexts;
import ru.yandex.qe.mail.meetings.rooms.dao.SentEmailsDao;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.EventSchedule;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Repetition;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resource;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Resources;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Response;
import ru.yandex.qe.mail.meetings.services.calendar.dto.WebEventData;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;
import ru.yandex.qe.mail.meetings.utils.DateRange;
import ru.yandex.qe.mail.meetings.utils.OfficeSplitter;

import static ru.yandex.qe.mail.meetings.utils.DateUtils.minusMinutes;

/**
 * Naive resource search "engine"
 *
 * @author Sergey Galyamichev
 */
@Component
public class SearchResourceTask {
    private static final Logger LOG = LoggerFactory.getLogger(SearchResourceTask.class);
    public static final String MOSCOW_TZ = "Europe/Moscow";

    private final CalendarWeb calendarWeb;
    private final CalendarUpdate calendarUpdate;
    private final StaffClient staffClient;
    private final CalendarHelper calendar;
    private final SentEmailsDao sentEmailsDao;
    private final OfficeSplitter officeSplitter;

    public SearchResourceTask(CalendarWeb calendarWeb, CalendarUpdate calendarUpdate, StaffClient staffClient, CalendarHelper calendar, SentEmailsDao sentEmailsDao, OfficeSplitter officeSplitter) {
        this.calendarWeb = calendarWeb;
        this.calendarUpdate = calendarUpdate;
        this.staffClient = staffClient;
        this.calendar = calendar;
        this.sentEmailsDao = sentEmailsDao;
        this.officeSplitter = officeSplitter;
    }

    /**
     * Request available resources for a time of the event and add if any
     *
     * @param action resource request details
     * @return true if resource found
     */
    Status executeFor(CalendarAction action, Date fireTime) {
        if (action.getStart().before(DateRange.toMSK(fireTime))) {
            return Status.OUTDATED; // no email notification will be sent
        }
        Contexts.Scan context = Contexts.Scan.parse(action.getContext());

        boolean applyToFuture = action.getInstanceStartTs() == null;
        Event event = applyToFuture ? calendar.getSeriesInstance(action.getEventId()) : calendarWeb.getEvent(action.getEventId());

        if (event.getStart().getTime() != action.getStart().getTime()) {
            updateTime(action, event);
        }

        Repetition repetition = applyToFuture ? event.getRepetition() : null;
        event.setInstanceStartTs(action.getInstanceStartTs());
        Person person = staffClient.getByLogin(AbstractMessageBuilder.getLogin(action.getEmail()));
        Set<Integer> officeIds = officeSplitter.officesToAdd(event, context, person);
        Set<Integer> assigned = new HashSet<>();

        for (Integer officeId : officeIds) {
            EventSchedule schedule = new EventSchedule(event.getStart(), event.getEnd(), repetition);
            Resources resources = calendarWeb.findAvailableResources(person.getUid(), officeId, context.getFilter(), MOSCOW_TZ, schedule);
            if (addResource(applyToFuture, event, resources.getResources(), officeId, context.isChange())) {
                assigned.add(officeId);
                // sequence has been changed - re-read
                event = applyToFuture ? calendar.getSeriesInstance(action.getEventId()) : calendarWeb.getEvent(action.getEventId());
            }
        }
        if (assigned.size() > 0) {
            LOG.info("assigned {}, to assign {}", assigned, officeIds);
            if (officeIds.size() != assigned.size()) {
                Set<Integer> idList = new HashSet<>(officeIds);
                idList.removeAll(assigned);
                reschedule(action, event, Contexts.scan(context.isAuto(), context.isChange(), idList, context.getFilter()));
            }
            return Status.PERFORMED;
        } else {
            if (officeIds.isEmpty()) {
                return Status.CHANGED;
            }
            return Status.ACCEPTED;
        }
    }

    private void updateTime(CalendarAction action, Event event) {
        int ttl = (int) TimeUnit.MILLISECONDS.toMinutes(DateRange.range(action.getTriggerTime(), action.getStart()).duration());
        action.setTriggerTime(minusMinutes(event.getStart(), ttl));
        action.setStart(event.getStart());
        if (action.getInstanceStartTs() != null) {
            action.setInstanceStartTs(event.getInstanceStartTs());
        }

        sentEmailsDao.updateActionTime(action);
    }

    private void reschedule(CalendarAction action, Event event, Contexts.Scan scan) {
        LOG.info("reschedule action {}, context {}", action, scan);
        CalendarAction newAction = action.copy();
        newAction.setSequence(event.getSequence());
        newAction.setContext(scan.toString());
        sentEmailsDao.insertActions(Collections.singleton(newAction));
    }

    private boolean addResource(boolean applyToFuture, Event event, List<Resource.Info> resources, int officeId, boolean isChange) {
        return resources.stream()
                .filter(r -> Resource.Type.ROOM.toString().equals(r.getType()))
                .map(r -> calendarWeb.getResourceInfo(r.getEmail()))
                .anyMatch(r -> tryUpdateEvent(event, r, applyToFuture, officeId, isChange));
    }

    private boolean tryUpdateEvent(Event event, Resource.Info resource, boolean applyToFuture, int officeId, boolean isChange) {
        LOG.info("Found resource {} for event {} and change {} in office {}", resource.getName(), event.getEventId(), isChange, officeId);
        if (isChange) {
            List<Resource.Info> resources = event.getResources().stream()
                    .filter(r -> r.getOfficeId() != officeId)
                    .collect(Collectors.toList());
            resources.add(resource);
            event.setResources(resources);
        } else {
            event.getResources().add(resource);
        }
        WebEventData request = WebEventData.fromEvent(event);
        Response response = calendarUpdate.updateEvent(event.getEventId(), event.getSequence(), event.getInstanceStartTs(), applyToFuture, true, null, request);
        if (!response.isOk()) {
            event.getResources().remove(resource);
        }
        return response.isOk();
    }
}
