package ru.yandex.qe.mail.meetings.rooms.dao;

import java.sql.Types;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Named;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.stereotype.Repository;

import ru.yandex.qe.mail.meetings.api.resource.dto.ActionType;
import ru.yandex.qe.mail.meetings.api.resource.dto.CalendarAction;
import ru.yandex.qe.mail.meetings.api.resource.dto.Status;
import ru.yandex.qe.mail.meetings.blamer.DeclineEvents;
import ru.yandex.qe.mail.meetings.utils.DateRange;

/**
 * @author Sergey Galyamichev
 */
@Repository
public class SentEmailsDao {
    private static final Logger LOG = LoggerFactory.getLogger(SentEmailsDao.class);

    public static final String ACTION_FIELDS = "id, status, sequence, email, type, start, instanceStartTs, event_id, create_date, trigger_time, context, action_id, group_id ";
    private NamedParameterJdbcTemplate jdbcTemplate;

    @Autowired
    public SentEmailsDao(@Named("psqlDataSource") DataSource psqlDataSource) {
        this.jdbcTemplate = new NamedParameterJdbcTemplate(psqlDataSource);
    }

    public Set<String> getSentEmails(DateRange lastWeek) {
        SqlParameterSource params = new MapSqlParameterSource("start_ts",  lastWeek.getFrom())
                .addValue("end_ts", lastWeek.getTo())
                .addValue("status", Status.SENT, Types.VARCHAR)
                .addValue("action_type", ActionType.NOTIFICATION, Types.VARCHAR);
        String sql =
                "SELECT " +
                    "status, email, event_id, type, context " +
                "FROM " +
                    "actions " +
                "WHERE " +
                    "status = :status AND type = :action_type AND start BETWEEN :start_ts AND :end_ts";
        return jdbcTemplate.query(sql, params, BeanPropertyRowMapper.newInstance(CalendarAction.class)).stream()
                .map(CalendarAction::getEmail)
                .collect(Collectors.toSet());
    }

    public boolean updateSent(DeclineEvents event, Date from) {
        SqlParameterSource params = new MapSqlParameterSource("email",  event.getEmail())
                .addValue("status", Status.SENT, Types.VARCHAR)
                .addValue("start", from)
                .addValue("action_type", ActionType.NOTIFICATION, Types.VARCHAR);
        String sql =
                "INSERT INTO " +
                    "actions(status, email, type, start) " +
                "VALUES " +
                    "(:status, :email, :action_type, :start)";
        return jdbcTemplate.update(sql, params) == 1;
    }

    public List<CalendarAction> getExpiredPendingActions(Date fireTime) {
        SqlParameterSource params = new MapSqlParameterSource("fireTime",  fireTime)
                .addValue("status", Status.PENDING, Types.VARCHAR);
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                    "status = :status AND trigger_time < :fireTime";
        return jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
    }

    public List<CalendarAction> getActions(Status status) {
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("status", status, Types.VARCHAR);
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                        "status = :status";
        return jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
    }

    public List<CalendarAction> getActions(ActionType type, Status status) {
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("type", type, Types.VARCHAR)
                .addValue("status", status, Types.VARCHAR);
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                        "status = :status AND type = :type";
        return jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
    }

    public List<CalendarAction> getActions(Integer eventId, ActionType type) {
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("type", type, Types.VARCHAR)
                .addValue("event_id", eventId, Types.INTEGER);
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                        "event_id = :event_id AND type = :type";
        return jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
    }

    public List<CalendarAction> getActions(String groupId) {
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("group_id", groupId, Types.VARCHAR);
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                        "group_id = :group_id";
        return jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
    }

    public List<CalendarAction> getActions(DateRange range) {
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("start_ts",  range.getFrom())
                .addValue("end_ts",  range.getTo());
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                        "start BETWEEN :start_ts AND :end_ts";
        return jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
    }

    public CalendarAction getAction(String actionId) {
        SqlParameterSource params = new MapSqlParameterSource()
                .addValue("action_id", actionId, Types.VARCHAR);
        String sql =
                "SELECT " +
                        ACTION_FIELDS +
                "FROM " +
                        "actions " +
                "WHERE " +
                        "action_id = :action_id";
        List<CalendarAction> action = jdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(CalendarAction.class));
        if (action.size() != 1) {
            LOG.warn("Found {} actions with id {}", action.size(), actionId);
        }
        return action.size() > 0 ? action.get(0) : null;
    }

    public void insertActions(Collection<? extends CalendarAction> actions) {
        SqlParameterSource[] params = actions.stream()
                .map(this::getBeanParameterSource)
                .toArray(SqlParameterSource[]::new);
        String sql =
                "INSERT INTO " +
                    "actions(status, sequence, email, type, start, instanceStartTs, event_id, create_date, trigger_time, context, action_id, group_id) " +
                "VALUES " +
                    "(:status, :sequence, :email, :type, :start, :instanceStartTs, :eventId, :createDate, :triggerTime, :context, :actionId, :groupId)";
        jdbcTemplate.batchUpdate(sql, params);
    }

    public void updateActionTime(CalendarAction action) {
        SqlParameterSource params = new MapSqlParameterSource("id", action.getId())
                .addValue("triggerTime", action.getTriggerTime())
                .addValue("startTime", action.getStart())
                .addValue("instanceStartTs", action.getInstanceStartTs());
        String sql =
                "UPDATE actions " +
                "SET " +
                        "trigger_time = :triggerTime, " +
                        "start = :startTime, " +
                        "instancestartts = :instanceStartTs " +
                "WHERE " +
                        "id = :id";
        jdbcTemplate.update(sql, params);
    }

    private BeanPropertySqlParameterSource getBeanParameterSource(CalendarAction action) {
        return new BeanPropertySqlParameterSource(action) {
            @Override
            public Object getValue(String paramName) throws IllegalArgumentException {
                Object result = super.getValue(paramName);
                return result instanceof Enum ? result.toString() : result;
            }
        };
    }

    public boolean updateAction(String actionId, Status status) {
        SqlParameterSource params = new MapSqlParameterSource("action_id", actionId)
                .addValue("status", status, Types.VARCHAR);
        String sql =
                "UPDATE " +
                    "actions " +
                "SET " +
                    "status = :status " +
                "WHERE " +
                    "action_id = :action_id";
        return jdbcTemplate.update(sql, params) == 1;
    }

    public boolean updateAction(String actionId, Status status, int sequence) {
        SqlParameterSource params = new MapSqlParameterSource("action_id", actionId)
                .addValue("sequence", sequence)
                .addValue("status", status, Types.VARCHAR);
        String sql =
                "UPDATE " +
                        "actions " +
                "SET " +
                        "status = :status " +
                "WHERE " +
                        "action_id = :action_id and sequence = :sequence";
        return jdbcTemplate.update(sql, params) == 1;
    }
}
