package ru.yandex.qe.mail.meetings.security.filters;

import java.io.IOException;

import javax.annotation.Nullable;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.http.HttpHeaders;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.filter.OncePerRequestFilter;
import org.springframework.web.util.WebUtils;

import ru.yandex.qe.mail.meetings.security.tokens.OAuthAuthenticationToken;
import ru.yandex.qe.mail.meetings.security.tokens.SessionAuthenticationToken;

/**
 * @author Sergey Galyamichev
 */
public class BlackBoxAuthenticationFilter extends OncePerRequestFilter {
    public static final String BLACKBOX = "blackbox";
    public static final String FORWARDED_FOR = "X-Forwarded-For";
    public static final String REAL_IP = "X-Real-IP";
    public static final String SESSION_ID = "Session_id";
    public static final String SESSION_ID_2 = "sessionid2";
    public static final String OAUTH_PREFIX = "OAuth ";

    @Override
    protected void doFilterInternal(HttpServletRequest request,
                                    HttpServletResponse response,
                                    FilterChain filterChain) throws ServletException, IOException
    {
        if (SecurityContextHolder.getContext().getAuthentication() == null) {
            SecurityContextHolder.getContext().setAuthentication(createAuthentication(request));
        }
        filterChain.doFilter(request, response);
    }

    @Nullable
    private Authentication createAuthentication(HttpServletRequest request) {
        String userIp = getUserIp(request);
        Cookie sessionIdCookie = WebUtils.getCookie(request, SESSION_ID);
        Cookie sslSessionIdCookie = WebUtils.getCookie(request, SESSION_ID_2);
        if (sessionIdCookie != null && sslSessionIdCookie != null) {
            logger.debug("Requesting authentication of " + userIp + "  by blackbox cookie");
            return new SessionAuthenticationToken(userIp, sessionIdCookie.getValue(), sslSessionIdCookie.getValue());
        }
        String authorizationHeader = request.getHeader(HttpHeaders.AUTHORIZATION);
        if (authorizationHeader != null && authorizationHeader.startsWith(OAUTH_PREFIX)) {
            logger.debug("Requesting authentication of " + userIp + " by blackbox token");
            return new OAuthAuthenticationToken(userIp, authorizationHeader.substring(OAUTH_PREFIX.length()));
        }
        return null;
    }

    public static String getUserIp(HttpServletRequest request) {
        String xForwardedFor = request.getHeader(FORWARDED_FOR);
        if (xForwardedFor != null) {
            return xForwardedFor.split(",", -1)[0];
        }
        String xRealIp = request.getHeader(REAL_IP);
        if (xRealIp != null) {
            return xRealIp;
        }
        return request.getRemoteAddr();
    }
}
