package ru.yandex.qe.mail.meetings.services;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.jaxrs.json.JacksonJaxbJsonProvider;
import org.apache.cxf.feature.Feature;
import org.apache.cxf.jaxrs.client.ClientConfiguration;
import org.apache.cxf.jaxrs.client.JAXRSClientFactory;
import org.apache.cxf.jaxrs.client.WebClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.qe.bus.features.log.LogFeature;
import ru.yandex.qe.logging.security.AuthorizationSecurityGuard;
import ru.yandex.qe.logging.security.CookieSecurityGuard;
import ru.yandex.qe.logging.security.PrivateHeaderSecurityGuard;
import ru.yandex.qe.logging.security.TvmSecurityGuard;
import ru.yandex.qe.mail.meetings.security.filters.BlackBoxAuthenticationFilter;
import ru.yandex.qe.mail.meetings.security.filters.TvmClientFilter;
import ru.yandex.qe.mail.meetings.services.abc.AbcApiV4;
import ru.yandex.qe.mail.meetings.services.abc.AbcClient;
import ru.yandex.qe.mail.meetings.services.blackbox.BlackBoxService;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.CommuneFaultInterceptor;
import ru.yandex.qe.mail.meetings.services.calendar.DateParameterConverterProvider;
import ru.yandex.qe.mail.meetings.services.calendar.UidClientFilter;
import ru.yandex.qe.mail.meetings.services.gaps.GapApi;
import ru.yandex.qe.mail.meetings.services.staff.StaffApiV3;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.tvm.TvmDaemonClient;

@Configuration
public class WebClientProvider {
    public static final String AUTHORIZATION_HEADER = "Authorization";
    public static final String OAUTH_FORMAT = "OAuth %s";

    @Value("${qe.app.name}")
    private String qeAppName;
    @Value("${calendar.hostname}")
    private String calendarHost;
    @Value("${calendar.update.hostname}")
    private String calendarUpdateHost;
    @Value("${staff.hostname}")
    private String staffHost;
    @Value("${gapapi.hostname}")
    private String gapHost;
    @Value("${staff.oauth.token}")
    private String oauthToken;
    @Value("${calendar.uid}")
    private String calendarUid;
    @Value("${abc.hostname}")
    private String abcHost;
    @Value("${tvm.token}")
    private String tvmToken;
    @Value("${tvm.hostname}")
    private String tvmHost;
    @Value("${blackbox.hostname}")
    private String blackboxHost;
    @Value("${tvm.enabled}")
    private boolean tvmEnabled;

    @Autowired
    private LogFeature logFeature;

    @Bean(name = "staffClient")
    public StaffClient getStaffClient() {
        List<?> providers = Collections.singletonList(new JacksonJaxbJsonProvider());
        StaffApiV3 api = JAXRSClientFactory.create(staffHost, StaffApiV3.class, providers,
                getLoggingFeatures(), null);
        WebClient.client(api).header(AUTHORIZATION_HEADER, String.format(OAUTH_FORMAT, oauthToken));
        return new StaffClient(api);
    }

    @Bean(name = "abcClient")
    public AbcClient getAbcClient() {
        List<?> providers = Collections.singletonList(new JacksonJaxbJsonProvider());
        AbcApiV4 api = JAXRSClientFactory.create(abcHost, AbcApiV4.class, providers,
                getLoggingFeatures(), null);
        WebClient.client(api).header(AUTHORIZATION_HEADER, String.format(OAUTH_FORMAT, oauthToken));
        return new AbcClient(api);
    }

    @Bean(name = "gapApi")
    public GapApi getGapApi() {
        List<?> providers = Collections.singletonList(new JacksonJaxbJsonProvider());
        GapApi api = JAXRSClientFactory.create(gapHost, GapApi.class, providers,
                getLoggingFeatures(), null);
        WebClient.client(api).header(AUTHORIZATION_HEADER, String.format(OAUTH_FORMAT, oauthToken));
        return api;
    }

    @Bean(name = "tvmDaemonClient")
    public TvmDaemonClient getTvmDaemonClient() {
        List<?> providers = Collections.singletonList(new JacksonJaxbJsonProvider());
        TvmDaemonClient api = JAXRSClientFactory.create(tvmHost, TvmDaemonClient.class, providers);
        WebClient.client(api).header(AUTHORIZATION_HEADER, tvmToken);
        ClientConfiguration config = WebClient.getConfig(api);
        config.getInInterceptors().add(new CommuneFaultInterceptor());
        return api;
    }

    @Bean(name = "blackBoxService")
    public BlackBoxService getBlackBoxService(TvmDaemonClient tvmDaemonClient) {
        List<Object> providers = new ArrayList<>(Collections.singletonList(new JacksonJaxbJsonProvider()));
        if (tvmEnabled) {
            providers.add(new TvmClientFilter(BlackBoxAuthenticationFilter.BLACKBOX, tvmDaemonClient, false));
        }
        return JAXRSClientFactory.create(blackboxHost, BlackBoxService.class, providers,
                getLoggingFeatures(), null);
    }

    @Bean(name = "calendarWeb")
    public CalendarWeb getCalendarWeb(TvmDaemonClient tvmDaemonClient) {
        List<Object> calendarProviders = getCalendarProviders(tvmDaemonClient);
        CalendarWeb api = JAXRSClientFactory.create(calendarHost, CalendarWeb.class, calendarProviders,
                getLoggingFeatures(), null);
        ClientConfiguration config = WebClient.getConfig(api);
        config.getInInterceptors().add(new CommuneFaultInterceptor());
        return api;
    }

    @Bean(name = "logFeature")
    public LogFeature getLogFeature() {
        final List<PrivateHeaderSecurityGuard> headerSecurityGuards = new ArrayList<>();
        headerSecurityGuards.add(new TvmSecurityGuard());
        headerSecurityGuards.add(new AuthorizationSecurityGuard());
        headerSecurityGuards.add(new CookieSecurityGuard());
        final LogFeature result = new LogFeature(qeAppName, headerSecurityGuards);
        result.setTraceLevel(true);
        return result;
    }

    private List<Feature> getLoggingFeatures() {
        return Collections.singletonList(logFeature);
    }

    @Bean(name = "calendarUpdate")
    public CalendarUpdate getCalendarUpdate(TvmDaemonClient tvmDaemonClient) {
        List<Object> calendarProviders = getCalendarProviders(tvmDaemonClient);
        CalendarUpdate api = JAXRSClientFactory.create(calendarUpdateHost, CalendarUpdate.class, calendarProviders,
                getLoggingFeatures(), null);
        ClientConfiguration config = WebClient.getConfig(api);
        config.getInInterceptors().add(new CommuneFaultInterceptor());
        return api;
    }

    private List<Object> getCalendarProviders(TvmDaemonClient tvmDaemonClient) {
        List<Object> providers = new ArrayList<>();
        providers.add(new JacksonJaxbJsonProvider());
        providers.add(new UidClientFilter(calendarUid));
        providers.add(new DateParameterConverterProvider());
        if (tvmEnabled) {
            providers.add(new TvmClientFilter("calendar", tvmDaemonClient, true));
        }
        return List.of(providers);
    }
}
