package ru.yandex.qe.mail.meetings.services.abc;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.mail.meetings.services.abc.dto.AbcCursorResponse;
import ru.yandex.qe.mail.meetings.services.abc.dto.AbcResponse;
import ru.yandex.qe.mail.meetings.services.abc.dto.AbcService;
import ru.yandex.qe.mail.meetings.services.abc.dto.ServiceMember;


public class AbcClient {
    private static final Logger LOG = LoggerFactory.getLogger(AbcClient.class);
    private static final String SERVICE_FIELDS = String.join(",", AbcService.ID, AbcService.SLUG, AbcService.NAME);
    private static final String MEMBERS_FIELDS = String.join(",", ServiceMember.PERSON_LOGIN);

    private static final int DEFAULT_LIMIT = 1000;

    @Nonnull
    private AbcApiV4 abcApiV4;

    public AbcClient(@Nonnull AbcApiV4 abcApiV4) {
        this.abcApiV4 = abcApiV4;
    }

    public List<AbcService> allServices() {
        LOG.debug("getAllAbcServices()");
        return allPage(page -> abcApiV4.services(SERVICE_FIELDS, DEFAULT_LIMIT, page));
    }

    public List<String> members(int serviceId) {
        LOG.debug("getAllmembers({})", serviceId);
        var result = allCursor(page -> abcApiV4.members(serviceId, MEMBERS_FIELDS, DEFAULT_LIMIT));
        return result.stream().map(ServiceMember::login).collect(Collectors.toList());
    }

    private <T> List<T> allPage(Function<Integer, AbcResponse<T>> requester) {
        List<T> result = null;
        int page = 1;
        AbcResponse<T> response;
        do {
            LOG.debug("apply page {}", page);
            response = requester.apply(page++);
            if (result == null) {
                result = new ArrayList<>(response.results());
            } else {
                result.addAll(response.results());
            }
        } while (page < response.pages());
        return result;
    }

    private <T> List<T> allCursor(Function<Integer, AbcCursorResponse<T>> requester) {
        List<T> result = null;
        int cursor = 0;
        String next = null;
        AbcCursorResponse<T> response;
        do {
            response = requester.apply(cursor);
            if (result == null) {
                result = new ArrayList<>(response.results());
            } else {
                result.addAll(response.results());
            }
            next = response.next();
            if (next != null) {
                throw new UnsupportedOperationException("cursor pagination was not supported yet");
            }
        } while (next != null);
        return result;
    }
}
