package ru.yandex.qe.mail.meetings.services.calendar;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.codec.Charsets;
import org.apache.cxf.helpers.IOUtils;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Message;
import org.apache.cxf.phase.AbstractPhaseInterceptor;
import org.apache.cxf.phase.Phase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.FaultResponse;

/**
 * Processes exception for calendar backend. Probably should be combined with Json -> Object conversion.
 *
 * @author Sergey Galiamichev
 */
public class CommuneFaultInterceptor extends AbstractPhaseInterceptor<Message> {
    private static final Logger LOG = LoggerFactory.getLogger(CommuneFaultInterceptor.class);
    private static final ObjectMapper OM = new ObjectMapper();
    public static final String WRAP_IN_PROCESSING_EXCEPTION = "wrap.in.processing.exception";

    public CommuneFaultInterceptor() {
        super(Phase.PRE_UNMARSHAL);
    }

    public void handleMessage(Message message) throws Fault {
        try {
            byte[] content = checkException(message);
            message.setContent(InputStream.class, new ByteArrayInputStream(content));
        } catch (IOException e) {
            LOG.error("Can't handle result", e);
        }
    }

    private byte[] checkException(Message message) throws IOException {
        InputStream inputStream = message.getContent(InputStream.class);
        byte[] content = IOUtils.readBytesFromStream(inputStream);
        JsonNode node = null;
        try {
            node = OM.readTree(content);
        } catch (Exception e) {
            LOG.error(String.format("Remote call read failed: %s", new String(content, Charsets.UTF_8)), e);
            throw e;
        }
        if (node != null && node.hasNonNull("error") && node.hasNonNull("invocationInfo")) {
            message.getExchange().put(WRAP_IN_PROCESSING_EXCEPTION, "false"); // re-throw my exception without wrapping
            FaultResponse response = OM.treeToValue(node, FaultResponse.class);
            String name = response.getError().getName();
            LOG.error("Remote call failed: name {}, {}", name, response.getError().getStackTrace());
            throw new CalendarException(name, response.getError().getMessage());
        }
        return content;
    }
}
