package ru.yandex.qe.mail.meetings.services.staff;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.Cacheable;

import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.services.staff.dto.PersonContainer;
import ru.yandex.qe.mail.meetings.services.staff.dto.Response;
import ru.yandex.qe.mail.meetings.services.staff.dto.StaffGroup;

@CacheConfig(cacheNames={"staff"})
public class StaffClient {
    private static final Logger LOG = LoggerFactory.getLogger(StaffClient.class);
    private static final String FIELDS = String.join(",", Person.LOGIN, Person.UID, Person.NAME, Person.EMAILS, Person.WORK_EMAIL);
    private static final String GROUP_FIELDS = String.join(",", StaffGroup.ID, StaffGroup.DESCRIPTION, StaffGroup.IS_DELETED, StaffGroup.NAME, StaffGroup.NAME, StaffGroup.URL);
    private static final String GROUP_MEMBERSHIP_FIELDS = String.join(",", "person");
    private static final String FIELDS_WITH_ACCOUNTS = String.join(",",
            Person.LOGIN, Person.UID, Person.NAME, Person.EMAILS, Person.ACCOUNTS, Person.LANGUAGE, Person.OFFICIAL, Person.LOCATION, Person.WORK_EMAIL);

    static final int DEFAULT_LIMIT = 1000;

    private StaffApiV3 staffApi;

    public StaffClient(StaffApiV3 staffApi) {
        this.staffApi = staffApi;
    }

    public List<Person> getAll() {
        LOG.debug("getAll()");
        return getAll(page -> staffApi.persons(FIELDS, DEFAULT_LIMIT, page));
    }

    @Cacheable
    public List<Person> getAll(boolean dismissed) {
        LOG.debug("getAll(dismissed = {})", dismissed);
        return getAll(page -> staffApi.persons(FIELDS, DEFAULT_LIMIT, page, dismissed));
    }

    public List<StaffGroup> getAllGroups() {
        LOG.debug("getAllGroups()");
        return getAll(page -> staffApi.groups(GROUP_FIELDS, DEFAULT_LIMIT, page));
    }

    public List<Person> groupMembers(int groupId, boolean showDismissed, boolean showRobots) {
        LOG.debug("groupMembership({})", groupId);
        return getAll(page -> staffApi.groupMembership(GROUP_MEMBERSHIP_FIELDS, DEFAULT_LIMIT, page, groupId))
                .stream()
                .map(PersonContainer::person)
                .filter(p -> showDismissed || !p.getOfficial().isDismissed())
                .filter(p -> showRobots || !p.getOfficial().isRobot())
                .collect(Collectors.toList());
    }

    @Cacheable
    public Person getByUid(String uid) {
        LOG.debug("In getByUid({})...", uid);
        List<Person> person = getAll(page -> staffApi.person(FIELDS, uid, null));
        if (person.isEmpty()) {
            return null;
        } else if (person.size() == 1) {
            return person.get(0);
        } else {
            throw new IllegalStateException("Got " + person);
        }
    }

    @Cacheable
    public Person getByLogin(String login) {
        LOG.debug("In getByLogin({})...", login);
        if (Objects.isNull(login)) {
            return null;
        }
        List<Person> person = getAll(page -> staffApi.person(FIELDS_WITH_ACCOUNTS, null, login));
        if (person.isEmpty()) {
            return null;
        } else if (person.size() == 1) {
            return person.get(0);
        } else {
            throw new IllegalStateException("Got " + person);
        }
    }

    public List<Person> getByLogins(Collection<String> logins, boolean dismissed) {
        LOG.debug("In getByLogin({})...", logins);
        return getAll(page -> staffApi.persons(FIELDS_WITH_ACCOUNTS, Integer.MAX_VALUE, page, dismissed, String.join(",", logins)));
    }

    private <T> List<T> getAll(Function<Integer, Response<T>> requester) {
        List<T> result = null;
        int page = 1;
        Response<T> response;
        do {
            response = requester.apply(page++);
            if (result == null) {
                result = new ArrayList<>(response.getTotal());
            }
            result.addAll(response.getResult());
            if (response.getPage() != page - 1) {
                throw new RuntimeException("page missmatch expected " + (page - 1) + " got " + response.getPage() + "!");
            }
        } while (response.getPage() < response.getPages());
        return result;
    }
}

