package ru.yandex.qe.mail.meetings.services.staff.dto;

import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public class Person {
    public static final String UID = "uid";
    public static final String LOGIN = "login";
    public static final String NAME = "name";
    public static final String EMAILS = "emails";
    public static final String ACCOUNTS = "accounts";
    public static final String CREATED_AT = "created_at";
    public static final String GUID = "guid";
    public static final String ID = "id";
    public static final String WORK_EMAIL = "work_email";
    public static final String WORK_PHONE = "work_phone";
    public static final String OFFICIAL = "official";
    public static final String LOCATION = "location";
    public static final String LANGUAGE = "language";

    private final String uid;
    private final String login;
    private final Name name;
    private final List<Account> accounts;
    private final List<Email> emails;
    private final Official official;
    private final Languages language;
    private final Location location;
    private final String workEmail;

    @JsonCreator
    public Person(@JsonProperty(UID) String uid,
                  @JsonProperty(LOGIN) String login,
                  @JsonProperty(NAME) Name name,
                  @JsonProperty(ACCOUNTS) List<Account> accounts,
                  @JsonProperty(EMAILS) List<Email> emails,
                  @JsonProperty(WORK_EMAIL) String workEmail,
                  @JsonProperty(OFFICIAL) Official official,
                  @JsonProperty(LANGUAGE) Languages language,
                  @JsonProperty(LOCATION) Location location
    ) {
        this.uid = uid;
        this.login = login;
        this.name = name;
        this.accounts = accounts;
        this.emails = emails;
        this.workEmail = workEmail;
        this.official = official;
        this.language = language;
        this.location = location;
    }

    public String getUid() {
        return uid;
    }

    public String getLogin() {
        return login;
    }

    public Name getName() {
        return name;
    }

    public List<Account> getAccounts() {
        return accounts;
    }

    public List<Email> getEmails() {
        return emails;
    }

    public String getWorkEmail() {
        return workEmail;
    }

    public Official getOfficial() {
        return official;
    }

    public Languages getLanguage() {
        return language;
    }

    public Location getLocation() {
        return location;
    }

    @JsonIgnore
    public String getEmail(Email.SourceType type) {
        return emails.stream().filter(e -> e.getSourceType() == type)
                .findAny()
                .map(Email::getAddress)
                .orElse(null);
    }

    @JsonIgnore
    public String getAssitant() {
        return accounts == null ? null : accounts.stream()
                .filter(a -> "assistent".equals(a.getType()))
                .map(Account::getValue)
                .findAny()
                .orElse(null);
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class Name {
        public static final String FIRST = "first";
        public static final String MIDDLE = "middle";
        public static final String LAST = "last";
        public static final String HIDDEN_MIDDLE = "hidden_middle";

        private final LocalizedString first;
        private final String middle;
        private final LocalizedString last;
        private final boolean hiddenMiddle;

        @JsonCreator
        public Name(@JsonProperty(FIRST) LocalizedString first,
                    @JsonProperty(MIDDLE) String middle,
                    @JsonProperty(LAST) LocalizedString last,
                    @JsonProperty(HIDDEN_MIDDLE) boolean hiddenMiddle
        ) {
            this.first = first;
            this.middle = middle;
            this.last = last;
            this.hiddenMiddle = hiddenMiddle;
        }

        public LocalizedString getFirst() {
            return first;
        }

        public String getMiddle() {
            return middle;
        }

        public LocalizedString getLast() {
            return last;
        }

        @Override
        public String toString() {
            return "Name{first: " + first + ", middle: " + middle + ", last: " + last + "}";
        }

        public String toString(Language language) {
            return first.get(language) + (hiddenMiddle ? " " : " " + middle + " ") + last.get(language);
        }

        @Override
        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (other == null || getClass() != other.getClass()) {
                return false;
            }

            Name name = (Name) other;
            return first.equals(name.first) && last.equals(name.last) && middle.equals(name.middle);
        }

        @Override
        public int hashCode() {
            return Objects.hash(first, middle, last);
        }
    }
}
