package ru.yandex.qe.mail.meetings.synchronizer;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.transaction.Transactional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import ru.yandex.qe.mail.meetings.booking.RoomService;
import ru.yandex.qe.mail.meetings.services.abc.dto.AbcService;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarUpdate;
import ru.yandex.qe.mail.meetings.services.calendar.CalendarWeb;
import ru.yandex.qe.mail.meetings.services.calendar.dto.Event;
import ru.yandex.qe.mail.meetings.services.calendar.dto.User;
import ru.yandex.qe.mail.meetings.services.calendar.dto.WebEventData;
import ru.yandex.qe.mail.meetings.services.calendar.dto.faults.CalendarException;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.services.staff.dto.Person;
import ru.yandex.qe.mail.meetings.services.staff.dto.StaffGroup;
import ru.yandex.qe.mail.meetings.synchronizer.dao.SyncDao;
import ru.yandex.qe.mail.meetings.synchronizer.dto.IdWithType;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SourceType;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SyncErrors;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SyncEvent;
import ru.yandex.qe.mail.meetings.synchronizer.impl.AbcServices;
import ru.yandex.qe.mail.meetings.synchronizer.impl.StaffGroups;
import ru.yandex.qe.mail.meetings.utils.CalendarHelper;
import ru.yandex.qe.mail.meetings.utils.YandexUtils;
import ru.yandex.qe.mail.meetings.ws.handlers.HandlerResult;
import ru.yandex.qe.mail.meetings.ws.sync.SyncHtmlBuilder;

import static ru.yandex.qe.mail.meetings.synchronizer.dto.SourceType.ABC;
import static ru.yandex.qe.mail.meetings.synchronizer.dto.SourceType.PERSON;
import static ru.yandex.qe.mail.meetings.synchronizer.dto.SourceType.STAFF;

@Service("meetingSyncronizer")
public class MeetingSynchronizer {
    /**
     * Класс позволяет добавлять staff/abc-группы и людей в списки участников встреч.
     * Нужен для поддержания актуального списка участников разного рода регкуляргых встреч, рассчитанных на отдет/группу или т.п.
     */

    private static final Logger LOG = LoggerFactory.getLogger(MeetingSynchronizer.class);
    private static final String FOOTER_MARK = "--------------SyncInfo--------------";

    @Nonnull
    private final AbcServices abcServices;

    @Nonnull
    private final StaffGroups staffGroups;

    @Nonnull
    private final StaffClient staffClient;

    @Nonnull
    private final SyncDao syncDao;

    @Nonnull
    private final CalendarUpdate calendarUpdate;

    @Nonnull
    private final RoomService roomService;

    @Nonnull
    private final CalendarHelper calendarHelper;

    @Nonnull
    private final CalendarWeb calendarWeb;

    @Nonnull
    private final SyncHtmlBuilder syncHtmlBuilder;

    @Nonnull
    private final String host;

    private final ExecutorService executor = Executors.newSingleThreadExecutor();

    @Inject
    public MeetingSynchronizer(
            @Nonnull AbcServices abcServices,
            @Nonnull StaffGroups staffGroups,
            @Nonnull StaffClient staffClient,
            @Nonnull SyncDao syncDao,
            @Nonnull CalendarUpdate calendarUpdate,
            @Nonnull RoomService roomService,
            @Nonnull CalendarHelper calendarHelper,
            @Nonnull CalendarWeb calendarWeb,
            @Nonnull SyncHtmlBuilder syncHtmlBuilder,
            @Nonnull @Value("${project.rout}") String host) {
        this.abcServices = abcServices;
        this.staffGroups = staffGroups;
        this.staffClient = staffClient;
        this.syncDao = syncDao;
        this.calendarUpdate = calendarUpdate;
        this.roomService = roomService;
        this.calendarHelper = calendarHelper;
        this.calendarWeb = calendarWeb;
        this.syncHtmlBuilder = syncHtmlBuilder;
        this.host = host;
    }


    /**
     * Привязать список участников события {eventId} к указанным группам
     *
     * @param eventId  - событие
     * @param owner    - инициатор (нужен для проверки прав)
     * @param services - abc - сервисы
     * @param groups   - грппы на стаффе
     * @param persons  - логины людей
     * @return - Враппер над ошибкой
     */
    @Transactional(Transactional.TxType.REQUIRED)
    public HandlerResult<SyncErrors> assign(@Nonnull Integer eventId,
                                            @Nonnull String owner,
                                            @Nonnull Collection<AbcService> services,
                                            @Nonnull Collection<StaffGroup> groups,
                                            @Nonnull Collection<Person> persons) {
        LOG.info("assigning {} to {}|{}|{} by {}", eventId, services, groups, persons, owner);

        if (!checkPermission(eventId, owner)) {
            return HandlerResult.error(SyncErrors.PERMISSION_DENIED);
        }

        // удалим старый маппинг, если он был
        syncDao.unassign(eventId);

        var idWithTypes = new HashSet<IdWithType>();

        idWithTypes.addAll(toIdWithTypeSet(services, AbcService::id, ABC));
        idWithTypes.addAll(toIdWithTypeSet(groups, StaffGroup::id, SourceType.STAFF));
        idWithTypes.addAll(toIdWithTypeSet(persons, Person::getLogin, PERSON));

        var syncEvent = new SyncEvent(owner, eventId);
        syncDao.assign(
                syncEvent,
                idWithTypes
        );
        executor.execute(() -> doSync(syncEvent));
        return HandlerResult.ok();
    }

    /**
     * Проверяет права, может ли человек редактировать событие
     *
     * @param eventId - событие
     * @param login   - человек
     * @return - хватает ли прав на редактирование
     */
    public boolean checkPermission(int eventId, String login) {
        var person = staffClient.getByLogin(login);
        var event = calendarWeb.getEvent(eventId, person.getUid());

        if (event.getActions().isEdit()) {
            return true;
        }

        if (login.equals(Optional.ofNullable(event.getOrganizer()).map(User::getLogin).orElse(null))) {
            return true;
        }

        var attendees = event.getAttendees().stream().map(User::getLogin).collect(Collectors.toSet());
        return event.isParticipantsCanInvite() && attendees.contains(login);
    }

    private <T> Set<IdWithType> toIdWithTypeSet(Collection<T> src, Function<T, ?> idFunc, SourceType type) {
        return src.stream()
                .map(idFunc)
                .map(id -> new IdWithType(id.toString(), type))
                .collect(Collectors.toSet());
    }

    /**
     * Актуализировать участников для события {eventId}
     *
     * @param syncEvent - событие
     */
    private void doSync(SyncEvent syncEvent) {
        LOG.info("sync on {} ({}) / {}", syncEvent.id(), syncEvent.owner(), Thread.currentThread().getName());
        final var eventId = syncEvent.id();

        try {
            // получаем аклуальный eventId
            final Event event;

            try {
                event = calendarHelper.getSeriesInstance(eventId);
            } catch (CalendarException e) {
                // если вылетел эксепшен с текстом 'event not found', то событие удалено
                if (e.getMessage() != null && e.getMessage().contains("event not found")) {
                    // отменяем синхронизацию удаленного события
                    syncDao.unassign(eventId);
                    return;
                }
                // это какой-то левый эксепшен, нужно его прокинуть наверх
                throw e;
            }

            var organizer = Optional.ofNullable(event.getOrganizer()).map(User::getLogin).orElse(null);

            // сформируем актуальный (новый) список участников
            var users = collectUsersForEvent(eventId, organizer);

            // реальный (текущий) список участников
            var currentUsers = Stream.concat(
                    Optional.ofNullable(event.getOrganizer()).map(User::getLogin).stream(),
                    event.getAttendees().stream().map(User::getLogin))
                    .collect(Collectors.toSet());

            // в случае различия ожиданий и реальности нужно заапдейтить событие
            if (!currentUsers.equals(users) || event.getDescription() == null || !event.getDescription().contains(FOOTER_MARK)) {
                var data = WebEventData.fromEvent(event);

                var description = createDescription(data, syncEvent);
                data.setDescription(description);


                // сохраняем забуканные переговорки
                var bookedRooms = data.getAttendees()
                        .stream()
                        .filter(p -> roomService.byMail(p).isPresent());

                // объединяем нужных людей с забуканными переговорками
                var usersAndRooms = Stream.concat(
                        users.stream(),
                        bookedRooms
                )
                        .map(YandexUtils::toYandexTeam)
                        .collect(Collectors.toList());

                data.setAttendees(usersAndRooms);

                LOG.info("updating event {} with users {}", eventId, usersAndRooms);

                // обновляем событие
                calendarUpdate.updateEvent(
                        event.getEventId(), event.getSequence(), event.getInstanceStartTs(), true, false, null, data
                );
            }
        } catch (Exception e) {
            if (e.getMessage() != null && e.getMessage().contains("event already modified")) {
                LOG.warn("event {} was modified, unassign", eventId);
                unassign(eventId);
            }
            LOG.error("cant sync event " + eventId, e);
        }
    }

    @Scheduled(cron = "${sync.cron.update.schedule}")
    public void syncAll() {
        var events = syncDao.getAllEvents();
        LOG.info("start synchronizing {} events", events.size());
        events.parallelStream().forEach(mute(this::doSync));
        LOG.info("sync finished");
    }

    public Map<SyncEvent, List<IdWithType>> getAllbyOwner(@Nonnull String login) {
        return syncDao.getAllByOwner(login);
    }

    public Map<SyncEvent, List<IdWithType>> getAllEventsWithLinkedGroups() {
        return syncDao.getAllEventsWithLinkedGroups();
    }

    public void unassign(int eventId) {
        LOG.info("canceling {}", eventId);
        syncDao.unassign(eventId);
        try {
            var event = calendarHelper.getSeriesInstance(eventId);
            var data = WebEventData.fromEvent(event);
            var desc = Optional.ofNullable(data.getDescription()).orElse("");
            if (desc.contains(FOOTER_MARK)) {
                desc = desc.split(FOOTER_MARK)[0];
                LOG.info("footer mark detected, new description: {}", desc);
            }
            data.setDescription(desc);
            calendarUpdate.updateEvent(
                    event.getEventId(), event.getSequence(), event.getInstanceStartTs(), true, false, null, data
            );
        } catch (Exception e) {
            LOG.warn("didn't change description after unassigning eventId: " + eventId, e);
        }
    }

    public void unassign(int eventId, IdWithType idWithType) {
        LOG.info("cancel {} - {}", eventId, idWithType);
        syncDao.unassign(eventId, idWithType);
        doSync(new SyncEvent("unknown-unassign", eventId));
    }

    private static <T> Consumer<T> mute(Consumer<T> consumer) {
        return x -> {
            try {
                consumer.accept(x);
            } catch (Exception e) {
                LOG.warn("muted error", e);
            }
        };
    }

    private String createDescription(WebEventData data, SyncEvent syncEvent) {
        var desc = Optional.ofNullable(data.getDescription()).orElse("");
        if (desc.contains(FOOTER_MARK)) {
            desc = desc.split(FOOTER_MARK)[0];
        }
        var assigned = syncDao.getAssignedIds(syncEvent.id());
        var descFooter =
                FOOTER_MARK +
                        "\n\n" +
                        "Участники встречи формируются автоматически\n\n" +
                        "\n\n" +
                        "Редактирование возможно через форму: https://forms.yandex-team.ru/surveys/28777" +
                        "\n\n" +
                        "Список участников:\n" +
                        syncHtmlBuilder.toPetty(syncEvent, assigned, false, ", ") +
                        "\n\n" +
                        SyncHtmlBuilder.cancelLink(syncEvent, host, "Отменить автоматическую синхронизацию");

        return desc + (desc.endsWith("\n") ? "" : '\n') + descFooter;
    }

    private Set<String> collectUsersForEvent(int eventId, String organizer) {
        // получаем идентификаторы объектов, из которых нужно сформировать список участников
        var assigned = syncDao.getAssignedIds(eventId);

        // получаем участников abc-групп
        var abcStream = extractLogins(assigned, ABC, Integer::valueOf, abcServices::membersById, false);
        // получаем участников staff-групп
        var staffStream = extractLogins(assigned, STAFF, Integer::valueOf, staffGroups::membersById, false);
        // добавляем индивидуальных участников
        var personStream = extractLogins(assigned, PERSON, Function.identity(), login -> Collections.singletonList(staffClient.getByLogin(login)), false);

        return Stream.of(
                abcStream,
                staffStream,
                personStream,
                Optional.ofNullable(organizer).stream()
        )
                .flatMap(s -> s)
                .collect(Collectors.toSet());
    }

    private <T> Stream<String> extractLogins(
            Collection<IdWithType> ids,
            SourceType type,
            Function<String, T> idCast,
            Function<T, List<Person>> members,
            boolean allowDismissed) {
        return ids
                .stream()
                .filter(withType(type))
                .map(id -> idCast.apply(id.id))
                .flatMap(id -> members.apply(id).stream())
                .filter(p -> allowDismissed || !p.getOfficial().isDismissed())
                .map(Person::getLogin);
    }

    private Predicate<IdWithType> withType(SourceType type) {
        return idWithType -> idWithType.type == type;
    }
}
