package ru.yandex.qe.mail.meetings.synchronizer;

import java.util.HashMap;
import java.util.Map;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import com.fasterxml.jackson.databind.ObjectMapper;
import freemarker.template.Configuration;
import org.apache.commons.lang3.CharEncoding;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.springframework.ui.freemarker.FreeMarkerTemplateUtils;

import ru.yandex.qe.mail.meetings.config.NotificationConfiguration;
import ru.yandex.qe.mail.meetings.cron.AbstractMessageBuilder;
import ru.yandex.qe.mail.meetings.services.staff.StaffClient;
import ru.yandex.qe.mail.meetings.synchronizer.dto.SyncErrors;


@Component
public class SyncResultMessageBuilder extends AbstractMessageBuilder {
    private static final Logger LOG = LoggerFactory.getLogger(SyncResultMessageBuilder.class);

    private static final String TEMPLATE_OK = "sync_success.ftl";
    private static final String TEMPLATE_FAIL = "sync_error.ftl";

    private static final ObjectMapper MAPPER = new ObjectMapper();

    public SyncResultMessageBuilder(Configuration configuration, StaffClient staffClient, NotificationConfiguration notConfig) {
        super(configuration, staffClient, notConfig);
    }


    public void prepareOkMessage(MimeMessage mimeMessage, String user, Map<String, String> params) throws MessagingException {
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, CharEncoding.UTF_8);
        prepareMessageHeadersDirect(message, user + AT_YANDEX, null);
        message.setSubject(user + "@, синхронизация началась");
        message.setText(buildBody(TEMPLATE_OK, user, params, null, false), true);
    }

    public void prepareErrorMessage(MimeMessage mimeMessage, String user, Map<String, String> params, SyncErrors reason, boolean isDebug) throws MessagingException {
        MimeMessageHelper message = new MimeMessageHelper(mimeMessage, true, CharEncoding.UTF_8);
        prepareMessageHeadersDirect(message, user + AT_YANDEX, null);
        message.setSubject(user + "@, " + " синхронизация провалилась");
        message.setText(buildBody(TEMPLATE_FAIL, user, params, reason, isDebug), true);
    }

    private String buildBody(String template, String user, Map<String, String> params, SyncErrors reason, boolean isDebug) throws MessagingException {
        try {
            Map<String, Object> valuesMap = new HashMap<>();

            valuesMap.put("name", user + "@");
            valuesMap.put("login", user);
            valuesMap.put("readableReason", makeReadable(reason));

            valuesMap.put("abcServices", params.getOrDefault("assign_abc_services", "-"));
            valuesMap.put("staffGroups", params.getOrDefault("assign_staff_groups", "-"));
            valuesMap.put("personLogins", params.getOrDefault("assign_person_logins", "-"));
            valuesMap.put("calendarUrl", params.getOrDefault("calendar_url_sync", "-"));

            if (isDebug) {
                valuesMap.put("fullConfig", MAPPER.writeValueAsString(params));
            } else {
                valuesMap.put("fullConfig", "");
            }

            LOG.info("preparing template {} with params {}", template, valuesMap);

            return FreeMarkerTemplateUtils.processTemplateIntoString(configuration.getTemplate(template), valuesMap);
        } catch (Exception e) {
            throw new MessagingException("Unable to build body", e);
        }
    }

    private String makeReadable(SyncErrors reason) {
        if (reason == null) {
            return "причина не установлена, обязательно сообщите нам об этой проблеме";
        }
        switch (reason) {
            case PERMISSION_DENIED:
                return "У вас не хватает прав для редактирования встречи";
            case UNKNOWN:
                return "внутренняя ошибка сервиса";
            default:
                return "К этой ошибке не написали описание: " + reason;
        }
    }
}

